import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ReviewService {
  private logger = new Logger('ReviewService');

  constructor(private prisma: PrismaService) {}

  // রিভিউ সাবমিট করুন
  async submitReview(
    tenantId: string,
    productId: string,
    userId: string,
    input: {
      rating: number;
      title: string;
      content: string;
    }
  ) {
    // ইতিমধ্যে রিভিউ আছে কিনা চেক করুন
    const existing = await this.prisma.productReview.findUnique({
      where: { productId_userId: { productId, userId } }
    });

    if (existing) {
      throw new BadRequestException('You have already reviewed this product');
    }

    // ক্রয় যাচাই করুন
    const purchase = await this.prisma.order.findFirst({
      where: {
        userId,
        items: { some: { productId } },
        status: { in: ['DELIVERED', 'COMPLETED'] }
      }
    });

    const review = await this.prisma.productReview.create({
      data: {
        tenantId,
        productId,
        userId,
        rating: Math.max(1, Math.min(5, input.rating)),
        title: input.title,
        content: input.content,
        verified: !!purchase,
        status: 'pending' // মডারেশন অপেক্ষায়
      }
    });

    this.logger.log(`Review submitted: ${review.id}`);

    return { success: true, data: review };
  }

  // রিভিউ অনুমোদন করুন
  async approveReview(reviewId: string) {
    const review = await this.prisma.productReview.update({
      where: { id: reviewId },
      data: { status: 'approved' }
    });

    return { success: true, data: review };
  }

  // রিভিউ প্রত্যাখ্যান করুন
  async rejectReview(reviewId: string, reason: string) {
    const review = await this.prisma.productReview.update({
      where: { id: reviewId },
      data: { status: 'rejected', moderationNote: reason }
    });

    return { success: true, data: review };
  }

  // রিভিউ পান (রেটিং অনুযায়ী ফিল্টার)
  async getReviews(
    productId: string,
    filters?: {
      rating?: number; // 1-5
      verified?: boolean;
      sort?: 'recent' | 'helpful' | 'rating_high' | 'rating_low';
    }
  ) {
    const where: any = {
      productId,
      status: 'approved'
    };

    if (filters?.rating) {
      where.rating = filters.rating;
    }

    if (filters?.verified !== undefined) {
      where.verified = filters.verified;
    }

    const orderBy: any = {};

    switch (filters?.sort) {
      case 'helpful':
        orderBy.helpful = 'desc';
        break;
      case 'rating_high':
        orderBy.rating = 'desc';
        break;
      case 'rating_low':
        orderBy.rating = 'asc';
        break;
      default:
        orderBy.createdAt = 'desc';
    }

    const reviews = await this.prisma.productReview.findMany({
      where,
      orderBy,
      include: { user: { select: { name: true } } }
    });

    // রেটিং সংক্ষিপ্তসার
    const summary = {
      total: reviews.length,
      avgRating: reviews.length > 0
        ? (reviews.reduce((sum, r) => sum + r.rating, 0) / reviews.length).toFixed(1)
        : 0,
      distribution: {
        5: reviews.filter((r) => r.rating === 5).length,
        4: reviews.filter((r) => r.rating === 4).length,
        3: reviews.filter((r) => r.rating === 3).length,
        2: reviews.filter((r) => r.rating === 2).length,
        1: reviews.filter((r) => r.rating === 1).length
      }
    };

    return {
      reviews,
      summary
    };
  }

  // রিভিউ সহায়ক মার্ক করুন
  async markHelpful(reviewId: string) {
    const review = await this.prisma.productReview.update({
      where: { id: reviewId },
      data: { helpful: { increment: 1 } }
    });

    return { success: true, data: review };
  }
}