import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as fs from 'fs';

@Injectable()
export class PWAGeneratorService {
  private logger = new Logger('PWAGeneratorService');

  constructor(private prisma: PrismaService) {}

  // প্রতিটি স্টোরের জন্য PWA Manifest জেনারেট করুন
  async generateManifest(tenantId: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId },
      select: {
        name: true,
        logoUrl: true,
        description: true,
        theme: true
      }
    });

    if (!tenant) throw new Error('Tenant not found');

    const manifest = {
      name: tenant.name,
      short_name: tenant.name.substring(0, 12),
      description: tenant.description || 'Online Store',
      start_url: `/${tenantId}/`,
      scope: `/${tenantId}/`,
      display: 'standalone',
      orientation: 'portrait-primary',
      background_color: '#ffffff',
      theme_color: '#007bff',
      
      // ইনস্টলেশন সেটিংস
      install_prompt: true,
      shortcuts: [
        {
          name: 'Products',
          short_name: 'Products',
          description: 'View all products',
          url: `/${tenantId}/products`,
          icons: [
            {
              src: tenant.logoUrl || '/logo.png',
              sizes: '192x192'
            }
          ]
        },
        {
          name: 'Cart',
          short_name: 'Cart',
          description: 'View your cart',
          url: `/${tenantId}/cart`,
          icons: []
        }
      ],

      icons: [
        {
          src: tenant.logoUrl || '/logo-192.png',
          sizes: '192x192',
          type: 'image/png',
          purpose: 'any'
        },
        {
          src: tenant.logoUrl || '/logo-512.png',
          sizes: '512x512',
          type: 'image/png',
          purpose: 'any maskable'
        }
      ],

      // অফলাইন ফল্ব্যাক
      screenshots: [
        {
          src: '/screenshot-1.png',
          sizes: '540x720',
          type: 'image/png'
        }
      ]
    };

    return manifest;
  }

  // Service Worker জেনারেট করুন
  async generateServiceWorker(tenantId: string): Promise<string> {
    const sw = `
const CACHE_VERSION = '${tenantId}-v1';
const CACHE_NAME = \`\${CACHE_VERSION}-cache\`;

const URLS_TO_CACHE = [
  '/',
  '/styles.css',
  '/app.js',
  '/offline.html'
];

// Install event
self.addEventListener('install', (event) => {
  console.log('Service Worker installing...');
  event.waitUntil(
    caches.open(CACHE_NAME).then((cache) => {
      return cache.addAll(URLS_TO_CACHE);
    })
  );
  self.skipWaiting();
});

// Activate event
self.addEventListener('activate', (event) => {
  console.log('Service Worker activating...');
  event.waitUntil(
    caches.keys().then((cacheNames) => {
      return Promise.all(
        cacheNames.map((cacheName) => {
          if (cacheName !== CACHE_NAME) {
            console.log('Deleting old cache:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
  self.clients.claim();
});

// Fetch event - Network first, fallback to cache
self.addEventListener('fetch', (event) => {
  const { request } = event;

  // API requests - network first
  if (request.url.includes('/api/')) {
    event.respondWith(
      fetch(request)
        .then((response) => {
          if (response.ok) {
            const cache = caches.open(CACHE_NAME);
            cache.then((c) => c.put(request, response.clone()));
          }
          return response;
        })
        .catch(() => caches.match(request))
    );
  } else {
    // Static assets - cache first
    event.respondWith(
      caches.match(request).then((cachedResponse) => {
        return cachedResponse || fetch(request);
      })
    );
  }
});

// Background sync
self.addEventListener('sync', (event) => {
  if (event.tag === 'sync-orders') {
    event.waitUntil(syncOrders());
  }
});

async function syncOrders() {
  try {
    const response = await fetch('/api/orders/sync');
    return response.json();
  } catch (error) {
    console.error('Sync failed:', error);
  }
}
`;

    return sw;
  }

  // Offline Page
  async generateOfflineHTML(tenantId: string): Promise<string> {
    const html = `
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Offline</title>
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', 'Roboto', 'Oxygen',
        'Ubuntu', 'Cantarell', 'Fira Sans', 'Droid Sans', 'Helvetica Neue', sans-serif;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      min-height: 100vh;
      display: flex;
      align-items: center;
      justify-content: center;
      color: white;
      padding: 20px;
    }
    .container {
      text-align: center;
      max-width: 400px;
    }
    h1 { font-size: 2em; margin-bottom: 10px; }
    p { font-size: 1.1em; margin-bottom: 20px; opacity: 0.9; }
    .icon { font-size: 4em; margin-bottom: 20px; }
    button {
      background: white;
      color: #667eea;
      border: none;
      padding: 12px 30px;
      border-radius: 25px;
      font-size: 1em;
      cursor: pointer;
      font-weight: bold;
      margin-top: 20px;
      transition: transform 0.2s;
    }
    button:active { transform: scale(0.95); }
  </style>
</head>
<body>
  <div class="container">
    <div class="icon">📱</div>
    <h1>You're Offline</h1>
    <p>Check your internet connection and try again.</p>
    <p>Don't worry! Your cart is saved locally.</p>
    <button onclick="location.reload()">Retry</button>
  </div>
  <script>
    window.addEventListener('online', () => location.reload());
  </script>
</body>
</html>
`;

    return html;
  }
}