import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as fs from 'fs';

@Injectable()
export class PwaService {
  private logger = new Logger('PwaService');

  constructor(private prisma: PrismaService) {}

  // PWA Manifest জেনারেট করুন
  async generateManifest(tenantId: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId }
    });

    const theme = await this.prisma.storeTheme.findUnique({
      where: { tenantId }
    });

    const manifest = {
      name: tenant?.name || 'Store',
      short_name: tenant?.name?.slice(0, 12) || 'Store',
      description: 'Progressive Web App',
      start_url: '/',
      display: 'standalone',
      background_color: theme?.bgColor || '#ffffff',
      theme_color: theme?.primaryColor || '#007bff',
      icons: [
        {
          src: theme?.logoUrl || '/logo.png',
          sizes: '192x192',
          type: 'image/png'
        },
        {
          src: theme?.logoUrl || '/logo.png',
          sizes: '512x512',
          type: 'image/png',
          purpose: 'any maskable'
        }
      ]
    };

    return manifest;
  }

  // Service Worker জেনারেট করুন
  async generateServiceWorker(tenantId: string) {
    const sw = `
const CACHE_NAME = 'store-v1';
const URLS_TO_CACHE = [
  '/',
  '/index.html',
  '/styles.css',
  '/app.js'
];

self.addEventListener('install', (event) => {
  event.waitUntil(
    caches.open(CACHE_NAME).then((cache) => {
      return cache.addAll(URLS_TO_CACHE);
    })
  );
});

self.addEventListener('fetch', (event) => {
  event.respondWith(
    caches.match(event.request).then((response) => {
      return response || fetch(event.request);
    })
  );
});
`;

    return sw;
  }

  // Offline page
  async generateOfflinePage(tenantId: string) {
    const html = `
<!DOCTYPE html>
<html>
<head>
  <title>Offline</title>
  <style>
    body { font-family: Arial; text-align: center; padding: 2rem; }
    h1 { color: #333; }
    p { color: #666; }
  </style>
</head>
<body>
  <h1>You're Offline</h1>
  <p>Check your internet connection and try again.</p>
</body>
</html>
`;

    return html;
  }
}