import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class RoleManagerService {
  constructor(private prisma: PrismaService) {}

  // পূর্ব-নির্ধারিত রোল তৈরি করুন
  async initializeDefaultRoles(tenantId: string) {
    const defaultRoles = [
      {
        roleName: 'Admin',
        permissions: [
          'read:*',
          'write:*',
          'delete:*',
          'manage:users',
          'manage:roles'
        ]
      },
      {
        roleName: 'Editor',
        permissions: [
          'read:products',
          'write:products',
          'read:orders',
          'read:customers'
        ]
      },
      {
        roleName: 'Viewer',
        permissions: ['read:products', 'read:orders', 'read:analytics']
      },
      {
        roleName: 'Support',
        permissions: [
          'read:orders',
          'write:orders',
          'read:customers',
          'write:customers'
        ]
      }
    ];

    for (const role of defaultRoles) {
      await this.prisma.rolePermission.create({
        data: {
          tenantId,
          ...role
        }
      });
    }
  }

  // কাস্টম রোল তৈরি করুন
  async createCustomRole(
    tenantId: string,
    roleName: string,
    permissions: string[]
  ) {
    const role = await this.prisma.rolePermission.create({
      data: {
        tenantId,
        roleName,
        permissions,
        description: `Custom role: ${roleName}`
      }
    });

    return { success: true, roleId: role.id };
  }

  // ব্যবহারকারীকে রোল অ্যাসাইন করুন
  async assignRoleToUser(userId: string, tenantId: string, roleId: string) {
    const assignment = await this.prisma.userRole.create({
      data: {
        userId,
        tenantId,
        roleId
      }
    });

    return { success: true };
  }

  // ব্যবহারকারীর পারমিশন পান
  async getUserPermissions(userId: string, tenantId: string) {
    const userRole = await this.prisma.userRole.findUnique({
      where: { userId_tenantId: { userId, tenantId } },
      include: { role: true }
    });

    if (!userRole) return { permissions: [] };

    return { permissions: userRole.role.permissions };
  }

  // পারমিশন যাচাই করুন
  async hasPermission(
    userId: string,
    tenantId: string,
    permission: string
  ): Promise<boolean> {
    const permissions = await this.getUserPermissions(userId, tenantId);

    // ওয়াইল্ডকার্ড সাপোর্ট (read:* অথবা write:*)
    return (
      permissions.permissions.includes(permission) ||
      permissions.permissions.includes(`${permission.split(':')[0]}:*`) ||
      permissions.permissions.includes('*')
    );
  }

  // সব রোল পান
  async getRoles(tenantId: string) {
    return this.prisma.rolePermission.findMany({
      where: { tenantId }
    });
  }
}