import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class AIRecommendationService {
  private logger = new Logger('AIRecommendationService');

  constructor(private prisma: PrismaService) {}

  // ব্যক্তিগত সুপারিশ পান
  async getPersonalizedRecommendations(userId: string, limit = 5) {
    // ব্যবহারকারীর ক্রয় ইতিহাস পান
    const orders = await this.prisma.order.findMany({
      where: { userId, status: 'DELIVERED' },
      include: { items: { include: { product: true } } },
      take: 10
    });

    // পছন্দের ক্যাটেগরি খুঁজে বের করুন
    const categories = new Set<string>();
    orders.forEach((order) => {
      order.items.forEach((item) => {
        if (item.product.categoryId) {
          categories.add(item.product.categoryId);
        }
      });
    });

    // একই ক্যাটেগরির প্রোডাক্ট সুপারিশ করুন
    const recommendations = await this.prisma.product.findMany({
      where: {
        categoryId: { in: Array.from(categories) },
        id: {
          notIn: orders.flatMap((o) => o.items).map((i) => i.productId)
        }
      },
      take: limit
    });

    return { recommendations };
  }

  // "আপনার মতো ক্রেতারা এটাও কিনেছেন"
  async getFrequentlyBoughtTogether(productId: string) {
    const orders = await this.prisma.order.findMany({
      where: {
        items: { some: { productId } }
      },
      include: { items: true }
    });

    // সম্পর্কিত পণ্য খুঁজে বের করুন
    const relatedProducts: any = {};

    orders.forEach((order) => {
      order.items.forEach((item) => {
        if (item.productId !== productId) {
          relatedProducts[item.productId] = (relatedProducts[item.productId] || 0) + 1;
        }
      });
    });

    // শীর্ষ সম্পর্কিত পণ্য পান
    const topRelated = Object.entries(relatedProducts)
      .sort((a: any, b: any) => b[1] - a[1])
      .slice(0, 5)
      .map((item: any) => item[0]);

    const products = await this.prisma.product.findMany({
      where: { id: { in: topRelated } }
    });

    return { products };
  }
}