import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class RecommendationService {
  private logger = new Logger('RecommendationService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // AI-ভিত্তিক সুপারিশ জেনারেট করুন
  async getSmartRecommendations(
    userId: string,
    tenantId: string,
    limit = 5
  ) {
    try {
      // ব্যবহারকারীর ক্রয় ইতিহাস পান
      const orders = await this.prisma.order.findMany({
        where: { userId, status: 'DELIVERED' },
        include: { items: { include: { product: true } } },
        take: 10
      });

      if (orders.length === 0) {
        // ডিফল্ট সুপারিশ (ট্রেন্ডিং)
        return await this.getTrendingProducts(tenantId, limit);
      }

      // ক্রয় করা প্রোডাক্টের ক্যাটেগরি সংগ্রহ করুন
      const purchasedCategories = new Set<string>();
      orders.forEach((order) => {
        order.items.forEach((item) => {
          if (item.product.categoryId) {
            purchasedCategories.add(item.product.categoryId);
          }
        });
      });

      // একই ক্যাটেগরির অন্যান্য পণ্য সুপারিশ করুন
      const recommendations = await this.prisma.product.findMany({
        where: {
          tenantId,
          categoryId: { in: Array.from(purchasedCategories) },
          id: {
            notIn: orders
              .flatMap((o) => o.items)
              .map((i) => i.productId)
          }
        },
        take: limit
      });

      // সুপারিশ সংরক্ষণ করুন
      await this.prisma.productRecommendation.create({
        data: {
          tenantId,
          userId,
          recommendedProductIds: recommendations.map((p) => p.id),
          reason: 'viewed_similar',
          score: 0.85
        }
      });

      return recommendations;
    } catch (err) {
      this.logger.error('Recommendation generation failed:', err);
      return await this.getTrendingProducts(tenantId, limit);
    }
  }

  // ট্রেন্ডিং প্রোডাক্ট পান
  private async getTrendingProducts(tenantId: string, limit = 5) {
    return await this.prisma.product.findMany({
      where: { tenantId },
      orderBy: { views: 'desc' },
      take: limit
    });
  }

  // "ক্রয়ের সাথে প্রায়শই কেনা" সুপারিশ
  async getFrequentlyBoughtTogether(productId: string) {
    const orders = await this.prisma.order.findMany({
      where: {
        items: {
          some: { productId }
        }
      },
      include: { items: true }
    });

    // সম্পর্কিত পণ্য গণনা করুন
    const related: any = {};

    orders.forEach((order) => {
      order.items.forEach((item) => {
        if (item.productId !== productId) {
          related[item.productId] = (related[item.productId] || 0) + 1;
        }
      });
    });

    // শীর্ষ 5 সম্পর্কিত পণ্য পান
    const topRelated = Object.entries(related)
      .sort((a: any, b: any) => b[1] - a[1])
      .slice(0, 5)
      .map((item: any) => item[0]);

    const products = await this.prisma.product.findMany({
      where: { id: { in: topRelated } }
    });

    return products;
  }
}