import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CartRecoveryReportService {
  constructor(private prisma: PrismaService) {}

  // পরিত্যক্ত কার্ট রিপোর্ট জেনারেট করুন
  async generateAbandonedCartReport(tenantId: string) {
    const abandonedCarts = await this.prisma.abandonedCart.findMany({
      where: { recoveredAt: null },
      include: { user: true }
    });

    const totalAbandonedCarts = abandonedCarts.length;
    const totalAbandonedValue = abandonedCarts.reduce(
      (sum, cart) => sum + Number(cart.cartTotal),
      0
    );

    // রিকভার করা কার্ট
    const recoveredCarts = await this.prisma.abandonedCart.findMany({
      where: { recoveredAt: { not: null } }
    });

    const recoveryRate =
      totalAbandonedCarts > 0
        ? (recoveredCarts.length / totalAbandonedCarts) * 100
        : 0;

    // সবচেয়ে বেশি পরিত্যক্ত প্রোডাক্ট
    const abandonedProducts: any = {};

    abandonedCarts.forEach((cart) => {
      const items = cart.cartItems as any[];
      items.forEach((item) => {
        if (!abandonedProducts[item.productId]) {
          abandonedProducts[item.productId] = {
            productId: item.productId,
            productName: item.productName,
            abandonedCount: 0,
            totalValue: 0
          };
        }

        abandonedProducts[item.productId].abandonedCount++;
        abandonedProducts[item.productId].totalValue += item.price * item.quantity;
      });
    });

    const topAbandonedProducts = Object.values(abandonedProducts)
      .sort((a: any, b: any) => b.abandonedCount - a.abandonedCount)
      .slice(0, 10);

    return {
      success: true,
      data: {
        totalAbandonedCarts,
        totalAbandonedValue,
        recoveryRate,
        topAbandonedProducts,
        recommendations: [
          `Recover $${(totalAbandonedValue * (recoveryRate / 100)).toFixed(2)} with better targeting`,
          'Focus on abandoned products: ' +
            topAbandonedProducts
              .slice(0, 3)
              .map((p: any) => p.productName)
              .join(', ')
        ]
      }
    };
  }
}