import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ProductPerformanceReportService {
  constructor(private prisma: PrismaService) {}

  // প্রোডাক্ট পারফরম্যান্স রিপোর্ট জেনারেট করুন
  async generateProductPerformanceReport(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId },
      include: { reviews: true }
    });

    const reports = await Promise.all(
      products.map(async (product) => {
        // ভিউ কাউন্ট
        const views = await this.prisma.analyticsEvent.count({
          where: {
            tenantId,
            pageUrl: { contains: product.id }
          }
        });

        // পারচেজ কাউন্ট
        const purchases = await this.prisma.order.count({
          where: {
            tenantId,
            items: { some: { productId: product.id } }
          }
        });

        // রেভিনিউ
        const orderItems = await this.prisma.orderItem.findMany({
          where: { productId: product.id }
        });

        const revenue = orderItems.reduce(
          (sum, item) => sum + item.price * item.quantity,
          0
        );

        const conversionRate = views > 0 ? (purchases / views) * 100 : 0;

        // রিভিউ রেটিং
        const avgRating =
          product.reviews.length > 0
            ? product.reviews.reduce((sum, r) => sum + r.rating, 0) /
              product.reviews.length
            : null;

        // ট্রেন্ড নির্ধারণ করুন
        const trend = await this.determineTrend(product.id);

        return {
          productId: product.id,
          productName: product.title,
          views,
          purchases,
          revenue,
          conversionRate,
          avgRating,
          reviewCount: product.reviews.length,
          trend,
          recommendations: this.getRecommendations(
            views,
            purchases,
            revenue,
            conversionRate
          )
        };
      })
    );

    // সর্বোচ্চ পারফরমার
    const topPerformers = reports.sort(
      (a, b) => b.revenue - a.revenue
    ).slice(0, 10);

    // সর্বনিম্ন পারফরমার
    const underperformers = reports.sort(
      (a, b) => a.revenue - b.revenue
    ).slice(0, 5);

    return {
      success: true,
      data: {
        totalProducts: products.length,
        topPerformers,
        underperformers,
        allReports: reports
      }
    };
  }

  private async determineTrend(productId: string): Promise<string> {
    // গত ৭ দিন এবং আগের ৭ দিনের তুলনা
    const lastWeek = new Date();
    lastWeek.setDate(lastWeek.getDate() - 7);

    const previousWeek = new Date();
    previousWeek.setDate(previousWeek.getDate() - 14);

    const lastWeekSales = await this.prisma.order.count({
      where: {
        items: { some: { productId } },
        createdAt: { gte: lastWeek }
      }
    });

    const previousWeekSales = await this.prisma.order.count({
      where: {
        items: { some: { productId } },
        createdAt: { gte: previousWeek, lt: lastWeek }
      }
    });

    if (lastWeekSales > previousWeekSales) return 'up';
    if (lastWeekSales < previousWeekSales) return 'down';
    return 'stable';
  }

  private getRecommendations(
    views: number,
    purchases: number,
    revenue: number,
    conversionRate: number
  ): string[] {
    const recommendations: string[] = [];

    if (views > 1000 && conversionRate < 1) {
      recommendations.push(
        'High traffic but low conversion - consider optimizing product page'
      );
    }

    if (views < 100) {
      recommendations.push(
        'Low visibility - increase marketing spend or improve SEO'
      );
    }

    if (revenue < 100) {
      recommendations.push('Low revenue - consider discontinuing or repricing');
    }

    return recommendations;
  }
}