import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class SalesReportService {
  constructor(private prisma: PrismaService) {}

  // বিক্রয় রিপোর্ট জেনারেট করুন
  async generateSalesReport(
    tenantId: string,
    period: 'daily' | 'weekly' | 'monthly'
  ) {
    const reportDate = new Date();
    let startDate = new Date();

    if (period === 'daily') {
      startDate.setDate(startDate.getDate() - 1);
    } else if (period === 'weekly') {
      startDate.setDate(startDate.getDate() - 7);
    } else if (period === 'monthly') {
      startDate.setMonth(startDate.getMonth() - 1);
    }

    // অর্ডার ডেটা সংগ্রহ করুন
    const orders = await this.prisma.order.findMany({
      where: {
        tenantId,
        createdAt: { gte: startDate }
      },
      include: { items: true }
    });

    // মেট্রিক্স ক্যালকুলেট করুন
    const totalOrders = orders.length;
    const totalRevenue = orders.reduce((sum, o) => sum + o.total, 0);
    const totalItems = orders.reduce((sum, o) => sum + o.items.length, 0);
    const avgOrderValue = totalOrders > 0 ? totalRevenue / totalOrders : 0;

    // শীর্ষ পণ্য খুঁজুন
    const productSales: any = {};

    orders.forEach((order) => {
      order.items.forEach((item) => {
        if (!productSales[item.productId]) {
          productSales[item.productId] = {
            productId: item.productId,
            quantity: 0,
            revenue: 0
          };
        }

        productSales[item.productId].quantity += item.quantity;
        productSales[item.productId].revenue += item.price * item.quantity;
      });
    });

    const topProducts = Object.values(productSales)
      .sort((a: any, b: any) => b.revenue - a.revenue)
      .slice(0, 5);

    // রিপোর্ট সংরক্ষণ করুন
    const report = await this.prisma.salesReport.create({
      data: {
        tenantId,
        reportDate: new Date(),
        period,
        totalOrders,
        totalRevenue: BigInt(Math.round(totalRevenue)) as any,
        totalItems,
        avgOrderValue: BigInt(Math.round(avgOrderValue)) as any,
        topProducts
      }
    });

    return {
      success: true,
      data: {
        period,
        totalOrders,
        totalRevenue,
        totalItems,
        avgOrderValue,
        topProducts,
        reportId: report.id
      }
    };
  }

  // ট্রেন্ড বিশ্লেষণ করুন
  async analyzeSalesMetrics(tenantId: string) {
    const today = new Date();
    const yesterday = new Date(today.getTime() - 24 * 60 * 60 * 1000);
    const lastWeek = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
    const lastMonth = new Date(today.getTime() - 30 * 24 * 60 * 60 * 1000);

    const todayOrders = await this.prisma.order.aggregate({
      where: { tenantId, createdAt: { gte: today } },
      _sum: { total: true },
      _count: true
    });

    const yesterdayOrders = await this.prisma.order.aggregate({
      where: {
        tenantId,
        createdAt: { gte: yesterday, lt: today }
      },
      _sum: { total: true }
    });

    const weekOrders = await this.prisma.order.aggregate({
      where: {
        tenantId,
        createdAt: { gte: lastWeek }
      },
      _sum: { total: true }
    });

    const monthOrders = await this.prisma.order.aggregate({
      where: {
        tenantId,
        createdAt: { gte: lastMonth }
      },
      _sum: { total: true }
    });

    const dayOverDayGrowth =
      yesterdayOrders._sum.total > 0
        ? ((todayOrders._sum.total - yesterdayOrders._sum.total) /
            yesterdayOrders._sum.total) *
          100
        : 0;

    return {
      today: {
        orders: todayOrders._count,
        revenue: todayOrders._sum.total
      },
      week: {
        revenue: weekOrders._sum.total
      },
      month: {
        revenue: monthOrders._sum.total
      },
      growth: {
        dayOverDay: dayOverDayGrowth
      }
    };
  }
}