import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import * as nodemailer from 'nodemailer';

@Injectable()
export class ScheduledReportService {
  private logger = new Logger('ScheduledReportService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // প্রতি মিনিটে শিডিউল করা রিপোর্ট চেক করুন
  @Cron('* * * * *')
  async checkAndSendScheduledReports() {
    const now = new Date();

    const scheduledReports = await this.prisma.scheduledReport.findMany({
      where: {
        isActive: true,
        nextScheduledAt: { lte: now }
      }
    });

    for (const report of scheduledReports) {
      try {
        await this.sendScheduledReport(report);

        // পরবর্তী রান নির্ধারণ করুন
        const nextRun = this.calculateNextRun(report);
        await this.prisma.scheduledReport.update({
          where: { id: report.id },
          data: {
            lastSentAt: new Date(),
            nextScheduledAt: nextRun
          }
        });

        this.logger.log(`Scheduled report sent: ${report.reportType}`);
      } catch (err) {
        this.logger.error(`Scheduled report failed: ${report.id}:`, err);
      }
    }
  }

  private async sendScheduledReport(report: any) {
    const emailContent = await this.generateReportEmail(
      report.tenantId,
      report.reportType
    );

    for (const email of report.recipientEmails) {
      await this.mailer.sendMail({
        from: process.env.STORE_EMAIL || 'noreply@store.com',
        to: email,
        subject: `${report.reportType.toUpperCase()} Report - ${new Date().toLocaleDateString()}`,
        html: emailContent
      });
    }
  }

  private async generateReportEmail(tenantId: string, reportType: string) {
    if (reportType === 'sales') {
      return await this.generateSalesReportEmail(tenantId);
    } else if (reportType === 'product') {
      return await this.generateProductReportEmail(tenantId);
    } else if (reportType === 'affiliate') {
      return await this.generateAffiliateReportEmail(tenantId);
    }

    return '<p>Report generated</p>';
  }

  private async generateSalesReportEmail(tenantId: string) {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    const orders = await this.prisma.order.findMany({
      where: {
        tenantId,
        createdAt: { gte: today }
      }
    });

    const revenue = orders.reduce((sum, o) => sum + o.total, 0);

    return `
<h2>Daily Sales Report</h2>
<table style="border-collapse: collapse; width: 100%;">
  <tr style="background: #f0f0f0;">
    <th style="border: 1px solid #ddd; padding: 10px;">Metric</th>
    <th style="border: 1px solid #ddd; padding: 10px;">Value</th>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Orders</td>
    <td style="border: 1px solid #ddd; padding: 10px;">${orders.length}</td>
  </tr>
  <tr>
    <td style="border: 1px solid #ddd; padding: 10px;">Total Revenue</td>
    <td style="border: 1px solid #ddd; padding: 10px;">$${revenue.toFixed(2)}</td>
  </tr>
</table>
    `;
  }

  private async generateProductReportEmail(tenantId: string) {
    // Product performance email generation
    return '<h2>Product Performance Report</h2>';
  }

  private async generateAffiliateReportEmail(tenantId: string) {
    // Affiliate report email generation
    return '<h2>Affiliate Performance Report</h2>';
  }

  private calculateNextRun(report: any): Date {
    const next = new Date();

    if (report.frequency === 'daily') {
      next.setDate(next.getDate() + 1);
      next.setHours(report.hour, report.minute, 0);
    } else if (report.frequency === 'weekly') {
      const daysUntilNextRun =
        (report.dayOfWeek - next.getDay() + 7) % 7 || 7;
      next.setDate(next.getDate() + daysUntilNextRun);
      next.setHours(report.hour, report.minute, 0);
    } else if (report.frequency === 'monthly') {
      next.setMonth(next.getMonth() + 1);
      next.setDate(report.dayOfMonth);
      next.setHours(report.hour, report.minute, 0);
    }

    return next;
  }

  // শিডিউল করা রিপোর্ট তৈরি করুন
  async createScheduledReport(
    tenantId: string,
    reportType: string,
    frequency: 'daily' | 'weekly' | 'monthly',
    recipientEmails: string[],
    hour: number = 9,
    minute: number = 0,
    dayOfWeek?: number,
    dayOfMonth?: number
  ) {
    const nextRun = this.calculateNextScheduledTime(
      frequency,
      hour,
      minute,
      dayOfWeek,
      dayOfMonth
    );

    const report = await this.prisma.scheduledReport.create({
      data: {
        tenantId,
        reportType,
        frequency,
        hour,
        minute,
        dayOfWeek,
        dayOfMonth,
        recipientEmails,
        nextScheduledAt: nextRun
      }
    });

    return { success: true, data: report };
  }

  private calculateNextScheduledTime(
    frequency: string,
    hour: number,
    minute: number,
    dayOfWeek?: number,
    dayOfMonth?: number
  ): Date {
    const next = new Date();
    next.setHours(hour, minute, 0, 0);

    if (frequency === 'daily') {
      next.setDate(next.getDate() + 1);
    } else if (frequency === 'weekly' && dayOfWeek !== undefined) {
      const daysUntilNextRun = (dayOfWeek - next.getDay() + 7) % 7 || 7;
      next.setDate(next.getDate() + daysUntilNextRun);
    } else if (frequency === 'monthly' && dayOfMonth) {
      next.setMonth(next.getMonth() + 1);
      next.setDate(dayOfMonth);
    }

    return next;
  }
}