import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class SearchService {
  constructor(private prisma: PrismaService) {}

  // রিয়েল-টাইম অটোকমপ্লিট সার্চ
  async autocompleteSearch(tenantId: string, query: string, limit = 10) {
    if (query.length < 2) return { suggestions: [] };

    // সার্চ ইন্ডেক্স থেকে দ্রুত সার্চ
    const suggestions = await this.prisma.searchIndex.findMany({
      where: {
        tenantId,
        searchText: { search: query }
      },
      select: { productId: true },
      take: limit
    });

    const products = await this.prisma.product.findMany({
      where: {
        id: { in: suggestions.map((s) => s.productId) }
      },
      select: { id: true, title: true, price: true, images: true }
    });

    return {
      suggestions: products.map((p) => ({
        id: p.id,
        title: p.title,
        price: p.price,
        image: Array.isArray(p.images) ? p.images[0] : null
      }))
    };
  }

  // ফিল্টার সহ পণ্য সার্চ
  async searchProducts(
    tenantId: string,
    query: string,
    filters: {
      categoryId?: string;
      minPrice?: number;
      maxPrice?: number;
      tags?: string[];
      inStock?: boolean;
    },
    page = 1,
    limit = 20
  ) {
    const skip = (page - 1) * limit;

    const where: any = {
      tenantId,
      title: { contains: query, mode: 'insensitive' }
    };

    if (filters.categoryId) where.categoryId = filters.categoryId;
    if (filters.minPrice || filters.maxPrice) {
      where.price = {};
      if (filters.minPrice) where.price.gte = filters.minPrice;
      if (filters.maxPrice) where.price.lte = filters.maxPrice;
    }
    if (filters.inStock) where.stock = { gt: 0 };

    const [products, total] = await Promise.all([
      this.prisma.product.findMany({
        where,
        skip,
        take: limit,
        include: { category: true }
      }),
      this.prisma.product.count({ where })
    ]);

    return {
      products,
      pagination: {
        page,
        limit,
        total,
        pages: Math.ceil(total / limit)
      }
    };
  }

  // উপলব্ধ ফিল্টার পান
  async getAvailableFilters(tenantId: string) {
    const categories = await this.prisma.productCategory.findMany({
      where: { tenantId },
      select: { id: true, name: true }
    });

    const priceRange = await this.prisma.product.aggregate({
      where: { tenantId },
      _min: { price: true },
      _max: { price: true }
    });

    return {
      categories,
      priceRange: {
        min: priceRange._min.price || 0,
        max: priceRange._max.price || 0
      }
    };
  }
}