import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as AWS from 'aws-sdk';

@Injectable()
export class ComplianceService {
  private logger = new Logger('ComplianceService');
  private s3 = new AWS.S3();

  constructor(private prisma: PrismaService) {}

  // ===== GDPR/PDPA COMPLIANCE =====
  async recordConsent(
    userId: string,
    consentType: string,
    isConsented: boolean
  ) {
    await this.prisma.dataConsent.create({
      data: {
        userId,
        consentType,
        isConsented,
        version: '1.0',
        consentedAt: isConsented ? new Date() : null
      }
    });

    return { success: true };
  }

  async getConsentStatus(userId: string) {
    return this.prisma.dataConsent.findMany({
      where: { userId }
    });
  }

  async deleteUserData(userId: string) {
    // GDPR অধিকার: ডেটা মুছে ফেলার অধিকার
    // কেবল অপরিহার্য ডেটা রাখুন

    await this.prisma.user.update({
      where: { id: userId },
      data: {
        name: 'Deleted User',
        email: `deleted-${userId}@example.com`,
        // সংবেদনশীল ডেটা মুছুন
      }
    });

    this.logger.log(`User data deleted: ${userId}`);

    return { success: true };
  }

  // ===== PASSWORD POLICY =====
  validatePassword(password: string): { valid: boolean; errors: string[] } {
    const errors: string[] = [];

    if (password.length < 8) {
      errors.push('Password must be at least 8 characters');
    }

    if (!/[A-Z]/.test(password)) {
      errors.push('Password must contain uppercase letters');
    }

    if (!/[a-z]/.test(password)) {
      errors.push('Password must contain lowercase letters');
    }

    if (!/[0-9]/.test(password)) {
      errors.push('Password must contain numbers');
    }

    if (!/[!@#$%^&*]/.test(password)) {
      errors.push('Password must contain special characters (!@#$%^&*)');
    }

    return {
      valid: errors.length === 0,
      errors
    };
  }

  async enforcePasswordChange(userId: string, daysAfter: number = 90) {
    // প্রতি ৯০ দিনে পাসওয়ার্ড পরিবর্তন করতে বাধ্য করুন
    // TODO: Implement password change reminder
  }

  // ===== API TOKEN EXPIRATION =====
  async expireOldTokens() {
    const now = new Date();

    await this.prisma.session.deleteMany({
      where: {
        expiresAt: { lt: now }
      }
    });

    this.logger.log('Old tokens expired');
  }

  // ===== SESSION TIMEOUT =====
  async checkSessionTimeout(sessionId: string, inactivityMinutes: number = 30) {
    const session = await this.prisma.session.findUnique({
      where: { id: sessionId }
    });

    if (!session) return true; // সেশন নেই

    const inactivityTime = Date.now() - new Date(session.createdAt).getTime();
    const timeoutMs = inactivityMinutes * 60 * 1000;

    return inactivityTime > timeoutMs;
  }

  // ===== BACKUP & RESTORE =====
  async createBackup(tenantId: string, countryCode?: string) {
    try {
      // সম্পূর্ণ ডেটা ডাম্প করুন
      const backupData = {
        tenantId,
        countryCode,
        timestamp: new Date(),
        // সব অপরিহার্য টেবিল
      };

      const key = `backups/${tenantId}/${countryCode || 'global'}/${Date.now()}.json`;

      await this.s3
        .putObject({
          Bucket: process.env.AWS_S3_BUCKET || '',
          Key: key,
          Body: JSON.stringify(backupData)
        })
        .promise();

      const backup = await this.prisma.backupRecord.create({
        data: {
          tenantId,
          countryCode,
          backupType: 'full',
          status: 'completed',
          backupUrl: `s3://${process.env.AWS_S3_BUCKET}/${key}`,
          size: BigInt(JSON.stringify(backupData).length),
          expiresAt: new Date(Date.now() + 90 * 24 * 60 * 60 * 1000) // ৯০ দিন
        }
      });

      this.logger.log(`Backup created: ${backup.id}`);

      return { success: true, backupId: backup.id };
    } catch (err) {
      this.logger.error('Backup creation failed:', err);
      throw err;
    }
  }

  async restoreBackup(backupId: string) {
    const backup = await this.prisma.backupRecord.findUnique({
      where: { id: backupId }
    });

    if (!backup) throw new Error('Backup not found');

    try {
      // S3 থেকে ডেটা ডাউনলোড করুন
      const data = await this.s3
        .getObject({
          Bucket: process.env.AWS_S3_BUCKET || '',
          Key: backup.backupUrl.replace('s3://', '').replace(`${process.env.AWS_S3_BUCKET}/`, '')
        })
        .promise();

      // ডেটা রিস্টোর করুন
      // TODO: Implement restore logic

      // রিকোয়েস্ট রেকর্ড করুন
      await this.prisma.restoreRequest.create({
        data: {
          tenantId: backup.tenantId,
          backupId,
          status: 'completed',
          requestedBy: 'system'
        }
      });

      this.logger.log(`Backup restored: ${backupId}`);

      return { success: true };
    } catch (err) {
      this.logger.error('Restore failed:', err);
      throw err;
    }
  }
}