import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as nodemailer from 'nodemailer';
import * as geoip from 'geoip-lite';

@Injectable()
export class SecurityService {
  private logger = new Logger('SecurityService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // ===== EMAIL VERIFICATION =====
  async sendVerificationEmail(email: string, userId: string) {
    const verificationToken = Math.random().toString(36).substring(2, 10);

    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL,
      to: email,
      subject: 'Verify Your Email Address',
      html: `
        <p>Please verify your email by clicking the link below:</p>
        <a href="${process.env.APP_URL}/verify?token=${verificationToken}">
          Verify Email
        </a>
      `
    });

    // টোকেন সংরক্ষণ করুন
    // TODO: Store in cache with expiry (15 minutes)

    return { success: true };
  }

  async verifyEmail(userId: string, token: string) {
    // টোকেন যাচাই করুন
    // TODO: Verify from cache

    await this.prisma.user.update({
      where: { id: userId },
      data: { emailVerified: true }
    });

    return { success: true };
  }

  // ===== GEO-BLOCKING =====
  async checkGeoBlocking(ipAddress: string, tenantId: string): Promise<boolean> {
    const geo = geoip.lookup(ipAddress);

    if (!geo) return false;

    const blockingRule = await this.prisma.geoBlockingRule.findFirst({
      where: {
        tenantId,
        countryCode: geo.country,
        isActive: true
      }
    });

    return !!blockingRule;
  }

  async setGeoBlockingRule(
    tenantId: string,
    countryCode: string,
    blockType: 'block' | 'require_2fa' | 'warn'
  ) {
    await this.prisma.geoBlockingRule.upsert({
      where: { tenantId_countryCode: { tenantId, countryCode } },
      create: {
        tenantId,
        countryCode,
        blockType,
        isActive: true
      },
      update: { blockType, isActive: true }
    });

    return { success: true };
  }

  // ===== RATE LIMITING =====
  async checkRateLimit(endpoint: string, ipAddress: string): Promise<boolean> {
    const config = await this.prisma.rateLimitConfig.findUnique({
      where: { endpoint }
    });

    if (!config) return true; // লিমিট নেই

    // Redis থেকে রিকোয়েস্ট কাউন্ট পান
    // TODO: Implement with Redis

    return true; // রেট লিমিট অতিক্রম করেনি
  }

  async blockIP(ipAddress: string, reason: string, durationHours: number = 24) {
    await this.prisma.iPBlacklist.upsert({
      where: { ipAddress },
      create: {
        ipAddress,
        reason,
        bannedUntil: new Date(Date.now() + durationHours * 60 * 60 * 1000)
      },
      update: {
        bannedUntil: new Date(Date.now() + durationHours * 60 * 60 * 1000)
      }
    });

    return { success: true };
  }

  // ===== ACTIVITY LOGGING =====
  async logActivity(
    userId: string,
    action: string,
    resource: string,
    resourceId: string,
    ipAddress: string,
    userAgent: string,
    status: 'success' | 'failed' = 'success'
  ) {
    const geo = geoip.lookup(ipAddress);

    await this.prisma.activityLog.create({
      data: {
        userId,
        action,
        resource,
        resourceId,
        ipAddress,
        userAgent,
        country: geo?.country,
        city: geo?.city,
        status
      }
    });

    this.logger.log(`Activity logged: ${action} on ${resource} by ${userId}`);
  }

  async getActivityLogs(userId: string, limit: number = 100) {
    return this.prisma.activityLog.findMany({
      where: { userId },
      orderBy: { createdAt: 'desc' },
      take: limit
    });
  }

  // ===== HTTPS ENFORCEMENT =====
  async enforceHTTPS(domain: string): Promise<boolean> {
    // HSTS হেডার সেট করুন
    // TODO: Configure in reverse proxy (Nginx, CloudFlare)
    return true;
  }
}