import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class LocalizedSEOService {
  constructor(private prisma: PrismaService) {}

  // লোকালাইজড মেটা ট্যাগ জেনারেট করুন
  async generateLocalizedMeta(
    entityType: string,
    entityId: string,
    countryCode: string
  ) {
    const content = await this.prisma.localizedContent.findUnique({
      where: {
        entityType_entityId_countryCode: {
          entityType,
          entityId,
          countryCode
        }
      }
    });

    if (!content) throw new Error('Localized content not found');

    return {
      title: content.metaTitle,
      description: content.metaDescription,
      keywords: this.generateKeywords(content.metaTitle, countryCode),
      hreflang: this.generateHrefLang(entityId, countryCode)
    };
  }

  // hreflang ট্যাগ জেনারেট করুন (মাল্টি-কান্ট্রি)
  private generateHrefLang(entityId: string, currentCountry: string): string[] {
    // সব সংস্করণের জন্য hreflang ট্যাগ
    const countries = ['BD', 'IN', 'US', 'GB']; // আপনার সমর্থিত দেশগুলি

    return countries.map((country) => {
      const locale = `${currentCountry.toLowerCase()}-${country}`;
      return `<link rel="alternate" hreflang="${locale}" href="https://platform.com/${country}/${entityId}" />`;
    });
  }

  // লোকাল কিওয়ার্ড জেনারেট করুন
  private generateKeywords(title: string, countryCode: string): string {
    // দেশ-নির্দিষ্ট কিওয়ার্ড সংযোজন করুন
    const countryKeywords: any = {
      BD: ['Bangladesh', 'Dhaka', 'বাংলাদেশ'],
      IN: ['India', 'Delhi', 'भारत'],
      US: ['USA', 'America', 'United States'],
      GB: ['UK', 'Britain', 'United Kingdom']
    };

    const baseKeywords = title.split(' ').slice(0, 3).join(', ');
    const local = countryKeywords[countryCode] || [];

    return `${baseKeywords}, ${local.join(', ')}`;
  }

  // সাইটম্যাপ জেনারেট করুন (দেশভিত্তিক)
  async generateLocalizedSitemap(countryCode: string) {
    const products = await this.prisma.product.findMany({
      where: { countryCode }
    });

    const sitemap = `<?xml version="1.0" encoding="UTF-8"?>
<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9"
        xmlns:xhtml="http://www.w3.org/1999/xhtml">
${products
  .map(
    (product) => `
  <url>
    <loc>https://platform.com/${countryCode}/${product.slug}</loc>
    <lastmod>${product.updatedAt.toISOString()}</lastmod>
    ${this.generateHrefLang(product.slug, countryCode).join('\n    ')}
  </url>
`
  )
  .join('')}
</urlset>`;

    return sitemap;
  }
}