import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class SEOScoreService {
  private logger = new Logger('SEOScoreService');

  constructor(private prisma: PrismaService) {}

  // প্রোডাক্টের SEO স্কোর ক্যালকুলেট করুন
  async calculateSEOScore(
    tenantId: string,
    pageUrl: string,
    pageData: any
  ): Promise<number> {
    let score = 0;
    const issues: any[] = [];

    // 1. Meta Title চেক (15 পয়েন্ট)
    if (pageData.metaTitle && pageData.metaTitle.length > 30 && pageData.metaTitle.length <= 60) {
      score += 15;
    } else {
      issues.push({
        type: 'meta_title',
        severity: 'high',
        suggestion: 'Meta title must be between 30-60 characters'
      });
    }

    // 2. Meta Description চেক (15 পয়েন্ট)
    if (pageData.metaDescription && pageData.metaDescription.length > 50 && pageData.metaDescription.length <= 160) {
      score += 15;
    } else {
      issues.push({
        type: 'meta_description',
        severity: 'high',
        suggestion: 'Meta description must be between 50-160 characters'
      });
    }

    // 3. H1 Tag চেক (10 পয়েন্ট)
    if (pageData.h1 && pageData.h1.length > 0) {
      score += 10;
    } else {
      issues.push({
        type: 'h1_tag',
        severity: 'high',
        suggestion: 'Add an H1 tag to your page'
      });
    }

    // 4. ইমেজ ALT টেক্সট (10 পয়েন্ট)
    if (pageData.images && pageData.images.every((img: any) => img.alt)) {
      score += 10;
    } else {
      issues.push({
        type: 'image_alt',
        severity: 'medium',
        suggestion: 'Add ALT text to all images'
      });
    }

    // 5. Internal Links (10 পয়েন্ট)
    if (pageData.internalLinks && pageData.internalLinks.length > 2) {
      score += 10;
    } else {
      issues.push({
        type: 'internal_links',
        severity: 'medium',
        suggestion: 'Add at least 3 internal links'
      });
    }

    // 6. Page Speed (10 পয়েন্ট)
    if (pageData.pageSpeed && pageData.pageSpeed < 3000) { // 3 সেকেন্ড
      score += 10;
    } else {
      issues.push({
        type: 'page_speed',
        severity: 'medium',
        suggestion: 'Improve page loading speed (target: < 3 seconds)'
      });
    }

    // 7. Mobile Responsive (10 পয়েন্ট)
    if (pageData.isMobileResponsive) {
      score += 10;
    } else {
      issues.push({
        type: 'mobile_responsive',
        severity: 'high',
        suggestion: 'Make your site mobile responsive'
      });
    }

    // 8. HTTPS (10 পয়েন্ট)
    if (pageUrl.startsWith('https://')) {
      score += 10;
    } else {
      issues.push({
        type: 'https',
        severity: 'high',
        suggestion: 'Use HTTPS for better security and SEO'
      });
    }

    // রেকর্ড সংরক্ষণ করুন
    await this.prisma.sEOScore.create({
      data: {
        tenantId,
        pageUrl,
        score: Math.min(score, 100),
        issues
      }
    });

    return Math.min(score, 100);
  }

  // SEO স্কোর পান
  async getSEOScore(pageUrl: string) {
    return this.prisma.sEOScore.findFirst({
      where: { pageUrl },
      orderBy: { recordedAt: 'desc' }
    });
  }
}