import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class SEOToolsService {
  private logger = new Logger('SEOToolsService');

  constructor(private prisma: PrismaService) {}

  // ===== IMAGE ALT AUTOMATION =====
  async generateImageAlt(imageUrl: string, productTitle: string): Promise<string> {
    // প্রোডাক্ট শিরোনাম থেকে ALT টেক্সট তৈরি করুন
    return `${productTitle} image`;
  }

  async bulkGenerateImageAlt(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId }
    });

    for (const product of products) {
      if (product.images && Array.isArray(product.images)) {
        const altTexts = product.images.map((img: any) =>
          this.generateImageAlt(img, product.title)
        );

        // আপডেট করুন
        // TODO: Update product with ALT texts
      }
    }

    return { success: true };
  }

  // ===== BULK META EDITOR =====
  async bulkUpdateMetaTags(
    tenantId: string,
    updates: { entityId: string; metaTitle: string; metaDescription: string }[]
  ) {
    for (const update of updates) {
      await this.prisma.metaTagConfig.updateMany({
        where: { tenantId, entityId: update.entityId },
        data: {
          metaTitle: update.metaTitle,
          metaDescription: update.metaDescription,
          isManual: true
        }
      });
    }

    this.logger.log(`Bulk updated ${updates.length} meta tags`);

    return { success: true, updated: updates.length };
  }

  // ===== BROKEN LINK CHECKER =====
  async checkBrokenLinks(tenantId: string) {
    const products = await this.prisma.product.findMany({
      where: { tenantId }
    });

    const brokenLinks = [];

    for (const product of products) {
      if (product.description) {
        const links = this.extractLinks(product.description);

        for (const link of links) {
          try {
            const response = await axios.head(link, { timeout: 5000 });

            if (response.status >= 400) {
              brokenLinks.push({
                sourceUrl: `/product/${product.id}`,
                targetUrl: link,
                statusCode: response.status
              });
            }
          } catch (err) {
            brokenLinks.push({
              sourceUrl: `/product/${product.id}`,
              targetUrl: link,
              statusCode: 404
            });
          }
        }
      }
    }

    // সংরক্ষণ করুন
    for (const brokenLink of brokenLinks) {
      await this.prisma.brokenLink.create({
        data: {
          tenantId,
          ...brokenLink
        }
      });
    }

    return {
      success: true,
      brokenLinksFound: brokenLinks.length,
      links: brokenLinks
    };
  }

  private extractLinks(text: string): string[] {
    const urlRegex = /(https?:\/\/[^\s]+)/g;
    return text.match(urlRegex) || [];
  }
}