import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class SetupWizardService {
  private logger = new Logger('SetupWizardService');

  constructor(private prisma: PrismaService) {}

  // সেটআপ শুরু করুন
  async startSetup(tenantId: string) {
    const setup = await this.prisma.storeSetup.upsert({
      where: { tenantId },
      create: {
        tenantId,
        step: 0,
        setupCompleted: false
      },
      update: {}
    });

    return { success: true, data: setup };
  }

  // ধাপ 1: ব্যবসায়িক তথ্য
  async step1_BusinessInfo(tenantId: string, input: {
    businessName: string;
    businessEmail: string;
    businessPhone: string;
    businessType: string;
  }) {
    const setup = await this.prisma.storeSetup.update({
      where: { tenantId },
      data: {
        step: 1,
        businessName: input.businessName,
        businessEmail: input.businessEmail,
        businessPhone: input.businessPhone,
        businessType: input.businessType
      }
    });

    return { success: true, data: setup };
  }

  // ধাপ 2: প্রথম পণ্য যোগ করুন
  async step2_AddProducts(tenantId: string, products: any[]) {
    // পণ্য যোগ করুন
    for (const product of products) {
      await this.prisma.product.create({
        data: {
          tenantId,
          sku: product.sku,
          title: product.title,
          price: product.price,
          stock: product.stock || 0
        }
      });
    }

    const setup = await this.prisma.storeSetup.update({
      where: { tenantId },
      data: {
        step: 2,
        hasProducts: true
      }
    });

    return { success: true, data: setup };
  }

  // ধাপ 3: পেমেন্ট সেটআপ
  async step3_PaymentSetup(tenantId: string, gatewayName: string) {
    const setup = await this.prisma.storeSetup.update({
      where: { tenantId },
      data: {
        step: 3,
        paymentSetup: true
      }
    });

    return { success: true, data: setup };
  }

  // ধাপ 4: শিপিং সেটআপ
  async step4_ShippingSetup(tenantId: string, couriers: string[]) {
    const setup = await this.prisma.storeSetup.update({
      where: { tenantId },
      data: {
        step: 4,
        shippingSetup: true
      }
    });

    return { success: true, data: setup };
  }

  // সেটআপ সম্পন্ন
  async completeSetup(tenantId: string) {
    const setup = await this.prisma.storeSetup.update({
      where: { tenantId },
      data: {
        step: 5,
        setupCompleted: true,
        completedAt: new Date()
      }
    });

    this.logger.log(`Store setup completed: ${tenantId}`);

    return { success: true, data: setup };
  }
}