import { Injectable, BadRequestException, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';
import * as crypto from 'crypto';

export interface ShippingQuote {
  courierId: string;
  courierName: string;
  cost: number;
  estimatedDays: number;
  currency: string;
}

export interface TrackingInfo {
  trackingNumber: string;
  status: string;
  currentLocation: string;
  events: any[];
  estimatedDelivery?: string;
}

@Injectable()
export class CourierService {
  private logger = new Logger('CourierService');

  constructor(private prisma: PrismaService) {}

  // Initialize all courier services (called on startup)
  async initializeCouriers() {
    const couriers = [
      // ===== SOUTH ASIA =====
      {
        name: 'Blue Dart',
        type: 'PRIVATE_COURIER',
        countries: ['IN', 'BD'],
        website: 'https://www.bluedart.com/',
        supportsTracking: true,
        baseCost: 50,
        costPerKg: 20
      },
      {
        name: 'TCS (Tranzum Courier Services)',
        type: 'PRIVATE_COURIER',
        countries: ['PK', 'AF'],
        website: 'https://www.tcs.com.pk/',
        supportsTracking: true,
        baseCost: 100,
        costPerKg: 25
      },
      {
        name: 'সুন্দরবন কুরিয়ার সার্ভিস',
        type: 'PRIVATE_COURIER',
        countries: ['BD'],
        website: 'https://sundarbancourier.com.bd/',
        supportsTracking: true,
        baseCost: 60,
        costPerKg: 15
      },
      {
        name: 'পাঠাও কুরিয়ার',
        type: 'LAST_MILE',
        countries: ['BD'],
        website: 'https://pathaocourier.com/',
        supportsTracking: true,
        baseCost: 40,
        costPerKg: 10
      },
      {
        name: 'Delhivery',
        type: 'PRIVATE_COURIER',
        countries: ['IN', 'BD'],
        website: 'https://www.delhivery.com/',
        supportsTracking: true,
        baseCost: 45,
        costPerKg: 18
      },
      {
        name: 'Leopard Courier Service',
        type: 'PRIVATE_COURIER',
        countries: ['PK'],
        website: 'https://www.leopardscourier.com/',
        supportsTracking: true,
        baseCost: 120,
        costPerKg: 28
      },
      {
        name: 'Nepal Post',
        type: 'NATIONAL_POST',
        countries: ['NP'],
        website: 'http://www.nepalpost.gov.np/',
        supportsTracking: true,
        baseCost: 50,
        costPerKg: 12
      },
      {
        name: 'Domex Express',
        type: 'PRIVATE_COURIER',
        countries: ['LK'],
        website: 'https://www.domex.lk/',
        supportsTracking: true,
        baseCost: 80,
        costPerKg: 20
      },

      // ===== EAST ASIA =====
      {
        name: 'SF Express',
        type: 'REGIONAL',
        countries: ['CN', 'TW', 'HK'],
        website: 'https://www.sf-express.com/',
        supportsTracking: true,
        baseCost: 100,
        costPerKg: 30
      },
      {
        name: 'Yamato Transport',
        type: 'PRIVATE_COURIER',
        countries: ['JP', 'TW'],
        website: 'https://www.kuronekoyamato.co.jp/',
        supportsTracking: true,
        baseCost: 150,
        costPerKg: 50
      },
      {
        name: 'CJ Logistics',
        type: 'PRIVATE_COURIER',
        countries: ['KR', 'CN', 'VN'],
        website: 'https://www.cjlogistics.com/',
        supportsTracking: true,
        baseCost: 120,
        costPerKg: 40
      },
      {
        name: 'Taiwan Post',
        type: 'NATIONAL_POST',
        countries: ['TW'],
        website: 'https://www.post.gov.tw/',
        supportsTracking: true,
        baseCost: 90,
        costPerKg: 25
      },
      {
        name: 'Hanjin Express',
        type: 'PRIVATE_COURIER',
        countries: ['KR'],
        website: 'https://www.hanjin.co.kr/',
        supportsTracking: true,
        baseCost: 110,
        costPerKg: 35
      },

      // ===== SOUTHEAST ASIA =====
      {
        name: 'JNE',
        type: 'PRIVATE_COURIER',
        countries: ['ID'],
        website: 'https://www.jne.co.id/',
        supportsTracking: true,
        baseCost: 80,
        costPerKg: 20
      },
      {
        name: 'Pos Laju',
        type: 'NATIONAL_POST',
        countries: ['MY'],
        website: 'https://www.pos.com.my/',
        supportsTracking: true,
        baseCost: 70,
        costPerKg: 18
      },
      {
        name: 'Kerry Express',
        type: 'REGIONAL',
        countries: ['TH', 'KH', 'LA', 'MY', 'MM'],
        website: 'https://th.kerryexpress.com/',
        supportsTracking: true,
        baseCost: 85,
        costPerKg: 22
      },
      {
        name: 'LBC Express',
        type: 'PRIVATE_COURIER',
        countries: ['PH'],
        website: 'https://www.lbcexpress.com/',
        supportsTracking: true,
        baseCost: 75,
        costPerKg: 20
      },
      {
        name: 'J&T Express',
        type: 'LAST_MILE',
        countries: ['ID', 'MY', 'PH'],
        website: 'https://www.jet.co.id/',
        supportsTracking: true,
        baseCost: 65,
        costPerKg: 15
      },
      {
        name: 'Ninja Van',
        type: 'LAST_MILE',
        countries: ['SG', 'MY', 'PH', 'TH', 'ID', 'VN'],
        website: 'https://www.ninjavan.co/',
        supportsTracking: true,
        baseCost: 70,
        costPerKg: 16
      },
      {
        name: 'VNPost',
        type: 'NATIONAL_POST',
        countries: ['VN'],
        website: 'https://www.vnpost.vn/',
        supportsTracking: true,
        baseCost: 60,
        costPerKg: 14
      },
      {
        name: 'Thailand Post',
        type: 'NATIONAL_POST',
        countries: ['TH'],
        website: 'https://www.thailandpost.co.th/',
        supportsTracking: true,
        baseCost: 75,
        costPerKg: 19
      },

      // ===== MIDDLE EAST =====
      {
        name: 'Aramex',
        type: 'REGIONAL',
        countries: ['SA', 'AE', 'QA', 'KW', 'BH', 'OM', 'YE', 'IQ', 'JO', 'LB', 'SY'],
        website: 'https://www.aramex.com/',
        supportsTracking: true,
        baseCost: 150,
        costPerKg: 45
      },
      {
        name: 'Emirates Post',
        type: 'NATIONAL_POST',
        countries: ['AE'],
        website: 'https://www.emiratespost.ae/',
        supportsTracking: true,
        baseCost: 120,
        costPerKg: 35
      },
      {
        name: 'Saudi Post / SPL',
        type: 'NATIONAL_POST',
        countries: ['SA'],
        website: 'https://splonline.com.sa/',
        supportsTracking: true,
        baseCost: 130,
        costPerKg: 40
      },

      // ===== CAUCASUS & CENTRAL ASIA =====
      {
        name: 'KazPost',
        type: 'NATIONAL_POST',
        countries: ['KZ'],
        website: 'https://www.kazpost.kz/',
        supportsTracking: true,
        baseCost: 100,
        costPerKg: 25
      },
      {
        name: 'Uzbekistan Post',
        type: 'NATIONAL_POST',
        countries: ['UZ'],
        website: 'https://www.pochta.uz/',
        supportsTracking: false,
        baseCost: 80,
        costPerKg: 20
      },

      // ===== INTERNATIONAL (Global) =====
      {
        name: 'DHL',
        type: 'INTERNATIONAL',
        countries: ['*'], // Global
        website: 'https://www.dhl.com/',
        supportsTracking: true,
        supportsPickup: true,
        baseCost: 200,
        costPerKg: 80
      },
      {
        name: 'FedEx',
        type: 'INTERNATIONAL',
        countries: ['*'], // Global
        website: 'https://www.fedex.com/',
        supportsTracking: true,
        supportsPickup: true,
        baseCost: 220,
        costPerKg: 90
      },
      {
        name: 'UPS',
        type: 'INTERNATIONAL',
        countries: ['*'], // Global
        website: 'https://www.ups.com/',
        supportsTracking: true,
        supportsPickup: true,
        baseCost: 210,
        costPerKg: 85
      }
    ];

    for (const courier of couriers) {
      await this.prisma.courierService.upsert({
        where: { name: courier.name },
        update: {},
        create: {
          ...courier,
          isActive: true
        }
      });
    }

    this.logger.log(`Initialized ${couriers.length} courier services`);
  }

  // Get available couriers for origin-destination pair
  async getAvailableCouriers(fromCountry: string, toCountry: string): Promise<CourierService[]> {
    const couriers = await this.prisma.courierService.findMany({
      where: {
        isActive: true,
        OR: [
          { countries: { has: fromCountry } },
          { countries: { has: toCountry } },
          { countries: { has: '*' } } // Global couriers
        ]
      },
      orderBy: { priority: 'desc' }
    });

    return couriers;
  }

  // Get shipping quotes from multiple couriers
  async getShippingQuotes(
    fromCountry: string,
    toCountry: string,
    weight: number, // kg
    dimensions?: { length: number; width: number; height: number }
  ): Promise<ShippingQuote[]> {
    const availableCouriers = await this.getAvailableCouriers(fromCountry, toCountry);

    const quotes: ShippingQuote[] = [];

    for (const courier of availableCouriers) {
      try {
        const rate = await this.prisma.shippingRate.findFirst({
          where: {
            courierId: courier.id,
            fromCountry,
            toCountry,
            weightMin: { lte: weight },
            weightMax: { gte: weight },
            isActive: true
          }
        });

        if (rate) {
          const cost = Number(rate.baseCost) + Number(rate.costPerKg) * weight;

          quotes.push({
            courierId: courier.id,
            courierName: courier.name,
            cost,
            estimatedDays: rate.estimatedDays,
            currency: 'USD'
          });
        }
      } catch (err) {
        this.logger.warn(`Failed to get quote from ${courier.name}`);
      }
    }

    return quotes.sort((a, b) => a.cost - b.cost);
  }

  // Create shipment with selected courier
  async createShipment(
    orderId: string,
    courierId: string,
    weight: number,
    pickupAddress: string,
    deliveryAddress: string
  ) {
    const order = await this.prisma.order.findUnique({
      where: { id: orderId }
    });

    if (!order) {
      throw new BadRequestException('Order not found');
    }

    const courier = await this.prisma.courierService.findUnique({
      where: { id: courierId }
    });

    if (!courier) {
      throw new BadRequestException('Courier not found');
    }

    // Generate tracking number
    const trackingNumber = this.generateTrackingNumber(courier.name);

    const shipment = await this.prisma.shipment.create({
      data: {
        orderId,
        courierId,
        trackingNumber,
        weight,
        pickupAddress,
        deliveryAddress,
        status: 'PENDING',
        cost: courier.baseCost || 0,
        currency: 'USD'
      }
    });

    this.logger.log(`Shipment created: ${trackingNumber}`);

    // Queue API call to courier to create actual shipment
    // TODO: Queue to BullMQ for async processing

    return {
      success: true,
      data: shipment
    };
  }

  // Get tracking information
  async getTracking(trackingNumber: string): Promise<TrackingInfo> {
    const shipment = await this.prisma.shipment.findUnique({
      where: { trackingNumber },
      include: { events: { orderBy: { timestamp: 'desc' } } }
    });

    if (!shipment) {
      throw new BadRequestException('Shipment not found');
    }

    return {
      trackingNumber: shipment.trackingNumber,
      status: shipment.status,
      currentLocation: shipment.currentLocation || 'Unknown',
      events: shipment.events,
      estimatedDelivery: shipment.estimatedDeliveryDate?.toISOString()
    };
  }

  // Generate shipping label (PDF)
  async generateShippingLabel(shipmentId: string): Promise<string> {
    const shipment = await this.prisma.shipment.findUnique({
      where: { id: shipmentId }
    });

    if (!shipment) {
      throw new BadRequestException('Shipment not found');
    }

    // Generate PDF label
    // TODO: Use pdfkit or similar to generate label
    const labelUrl = `/api/shipments/${shipmentId}/label.pdf`;

    await this.prisma.shipment.update({
      where: { id: shipmentId },
      data: { labelUrl }
    });

    return labelUrl;
  }

  // Webhook handler for courier updates
  async handleCourierWebhook(courierName: string, payload: any) {
    this.logger.log(`Webhook received from ${courierName}`);

    // Parse payload based on courier
    const trackingNumber = payload.tracking_number || payload.awb || payload.track_id;

    const shipment = await this.prisma.shipment.findUnique({
      where: { trackingNumber }
    });

    if (!shipment) {
      throw new BadRequestException('Shipment not found');
    }

    // Update shipment status
    const status = this.mapCourierStatusToShippingStatus(payload.status);

    await this.prisma.shipment.update({
      where: { id: shipment.id },
      data: {
        status,
        currentLocation: payload.location
      }
    });

    // Create event record
    await this.prisma.shippingEvent.create({
      data: {
        shipmentId: shipment.id,
        status,
        location: payload.location,
        message: payload.message
      }
    });

    // TODO: Send customer notification via email/SMS

    return { success: true };
  }

  private generateTrackingNumber(courierName: string): string {
    const prefix = courierName.slice(0, 3).toUpperCase();
    const randomStr = crypto.randomBytes(8).toString('hex').toUpperCase();
    return `${prefix}${randomStr}`;
  }

  private mapCourierStatusToShippingStatus(courierStatus: string): string {
    const statusMap: any = {
      'picked_up': 'PICKED_UP',
      'in_transit': 'IN_TRANSIT',
      'out_for_delivery': 'OUT_FOR_DELIVERY',
      'delivered': 'DELIVERED',
      'failed': 'FAILED',
      'returned': 'RETURNED'
    };

    return statusMap[courierStatus.toLowerCase()] || 'PENDING';
  }
}