import {
  Controller,
  Post,
  Get,
  Body,
  Param,
  Query,
  UseGuards,
  Req
} from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { CourierService } from './courier.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';

@ApiTags('Shipping')
@Controller('shipping')
export class ShippingController {
  constructor(private courierService: CourierService) {}

  @Get('couriers')
  async getAvailableCouriers(
    @Query('fromCountry') fromCountry: string,
    @Query('toCountry') toCountry: string
  ) {
    return {
      success: true,
      data: await this.courierService.getAvailableCouriers(fromCountry, toCountry)
    };
  }

  @Get('quotes')
  async getShippingQuotes(
    @Query('fromCountry') fromCountry: string,
    @Query('toCountry') toCountry: string,
    @Query('weight') weight: number
  ) {
    return {
      success: true,
      data: await this.courierService.getShippingQuotes(fromCountry, toCountry, weight)
    };
  }

  @Post('shipment')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async createShipment(
    @Req() req: any,
    @Body()
    body: {
      orderId: string;
      courierId: string;
      weight: number;
      pickupAddress: string;
      deliveryAddress: string;
    }
  ) {
    return await this.courierService.createShipment(
      body.orderId,
      body.courierId,
      body.weight,
      body.pickupAddress,
      body.deliveryAddress
    );
  }

  @Get('track/:trackingNumber')
  async trackShipment(@Param('trackingNumber') trackingNumber: string) {
    return {
      success: true,
      data: await this.courierService.getTracking(trackingNumber)
    };
  }

  @Get('label/:shipmentId')
  @UseGuards(JwtAuthGuard)
  @ApiBearerAuth()
  async getShippingLabel(@Param('shipmentId') shipmentId: string) {
    return {
      success: true,
      data: { labelUrl: await this.courierService.generateShippingLabel(shipmentId) }
    };
  }

  @Post('webhook/:courierName')
  async handleWebhook(
    @Param('courierName') courierName: string,
    @Body() body: any
  ) {
    return await this.courierService.handleCourierWebhook(courierName, body);
  }
}