import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class AIInsightsService {
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // ট্রাফিক বনাম বিক্রয় বিশ্লেষণ
  async analyzeTrafficVsSales() {
    const last30Days = new Date();
    last30Days.setDate(last30Days.getDate() - 30);

    // ট্রাফিক ডেটা
    const events = await this.prisma.analyticsEvent.findMany({
      where: { createdAt: { gte: last30Days } },
      take: 10000
    });

    // অর্ডার ডেটা
    const orders = await this.prisma.order.findMany({
      where: { createdAt: { gte: last30Days } }
    });

    const analysisPrompt = `
Analyze this e-commerce data:
- Total visitors (last 30 days): ${events.length}
- Total orders: ${orders.length}
- Total revenue: $${orders.reduce((sum, o) => sum + o.total, 0)}

Provide insights on:
1. Conversion efficiency
2. Performance bottlenecks
3. Opportunities for improvement

Format as JSON with keys: summary, bottlenecks, opportunities, recommendations
`;

    const response = await this.openai.chat.completions.create({
      model: 'gpt-3.5-turbo',
      messages: [{ role: 'user', content: analysisPrompt }],
      temperature: 0.7,
      max_tokens: 1000
    });

    try {
      return JSON.parse(response.choices[0].message.content || '{}');
    } catch {
      return { summary: response.choices[0].message.content };
    }
  }
}