import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class AutoUpgradeService {
  private logger = new Logger('AutoUpgradeService');

  constructor(private prisma: PrismaService) {}

  // প্রতিদিন আপগ্রেড চেক করুন
  @Cron('0 2 * * *')
  async checkAndAutoUpgradeTenants() {
    this.logger.log('Checking for auto-upgrades...');

    const tenants = await this.prisma.tenant.findMany({
      include: { subscription: true }
    });

    for (const tenant of tenants) {
      // ব্যবহার সীমা চেক করুন
      const usage = await this.getTenantUsage(tenant.id);

      if (
        usage.percentageUsed > 90 &&
        tenant.subscription?.plan !== 'enterprise'
      ) {
        // পরবর্তী পরিকল্পনায় আপগ্রেড করুন
        await this.upgradeTenant(
          tenant.id,
          'usage_exceeded',
          tenant.subscription?.plan || 'starter'
        );
      }
    }
  }

  private async upgradeTenant(
    tenantId: string,
    reason: string,
    fromPlan: string
  ) {
    const upgradePaths: any = {
      starter: 'professional',
      professional: 'enterprise'
    };

    const toPlan = upgradePaths[fromPlan] || 'enterprise';

    // সাবস্ক্রিপশন আপডেট করুন
    await this.prisma.subscription.updateMany({
      where: { tenantId },
      data: { plan: toPlan }
    });

    // লগ করুন
    await this.prisma.tenantAutoUpgradeLog.create({
      data: {
        tenantId,
        fromPlan,
        toPlan,
        reason
      }
    });

    this.logger.log(
      `Tenant auto-upgraded: ${tenantId} from ${fromPlan} to ${toPlan}`
    );
  }

  private async getTenantUsage(tenantId: string) {
    // ব্যবহার শতাংশ গণনা করুন
    const resourceUsage = await this.prisma.databaseUsage.findFirst({
      where: { tenantId },
      orderBy: { recordedAt: 'desc' }
    });

    if (!resourceUsage) return { percentageUsed: 0 };

    const percentageUsed =
      (Number(resourceUsage.storageUsed) / Number(resourceUsage.storageLimit)) *
      100;

    return { percentageUsed };
  }
}