import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CommunicationsService {
  private logger = new Logger('CommunicationsService');

  constructor(private prisma: PrismaService) {}

  // ===== COUNTRY PERFORMANCE =====
  async getCountryPerformanceComparison() {
    return this.prisma.countryPerformanceComparison.findMany({
      orderBy: { totalRevenue: 'desc' }
    });
  }

  // ===== ADMIN CHAT =====
  async sendMessageToTenant(
    adminId: string,
    tenantId: string,
    message: string
  ) {
    const msg = await this.prisma.adminChatMessage.create({
      data: {
        fromAdminId: adminId,
        toTenantId: tenantId,
        message
      }
    });

    this.logger.log(`Message sent to tenant ${tenantId}`);

    return { success: true, messageId: msg.id };
  }

  async getTenantMessages(tenantId: string) {
    return this.prisma.adminChatMessage.findMany({
      where: { toTenantId: tenantId },
      orderBy: { createdAt: 'desc' }
    });
  }

  // ===== ANNOUNCEMENTS =====
  async createAnnouncement(
    title: string,
    content: string,
    type: string,
    targetTenants?: string[]
  ) {
    const announcement = await this.prisma.adminAnnouncement.create({
      data: {
        title,
        content,
        type,
        targetTenants: targetTenants || [],
        isPublished: false
      }
    });

    return { success: true, announcementId: announcement.id };
  }

  async publishAnnouncement(announcementId: string) {
    const announcement = await this.prisma.adminAnnouncement.update({
      where: { id: announcementId },
      data: {
        isPublished: true,
        publishedAt: new Date()
      }
    });

    this.logger.log(`Announcement published: ${announcement.title}`);

    // সব টেন্যান্টকে বিজ্ঞপ্তি পাঠান
    // TODO: Send notifications

    return { success: true };
  }

  // ===== PLUGIN APPROVAL =====
  async getPendingPluginApprovals() {
    return this.prisma.pluginApprovalQueue.findMany({
      where: { status: 'pending' },
      orderBy: { submittedAt: 'desc' }
    });
  }

  async approvePlugin(queueId: string, reviewedBy: string) {
    await this.prisma.pluginApprovalQueue.update({
      where: { id: queueId },
      data: {
        status: 'approved',
        reviewedBy,
        reviewedAt: new Date()
      }
    });

    this.logger.log(`Plugin approved: ${queueId}`);

    return { success: true };
  }

  async rejectPlugin(
    queueId: string,
    reviewedBy: string,
    feedback: string
  ) {
    await this.prisma.pluginApprovalQueue.update({
      where: { id: queueId },
      data: {
        status: 'rejected',
        reviewedBy,
        reviewedAt: new Date(),
        feedback
      }
    });

    this.logger.log(`Plugin rejected: ${queueId}`);

    return { success: true };
  }
}