import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class GlobalAnalyticsService {
  private logger = new Logger('GlobalAnalyticsService');

  constructor(private prisma: PrismaService) {}

  // গ্লোবাল এনালিটিক্স পান
  async getGlobalAnalytics() {
    const today = new Date();
    today.setHours(0, 0, 0, 0);

    // মোট টেন্যান্ট
    const totalTenants = await this.prisma.tenant.count();
    const activeTenants = await this.prisma.tenant.count({
      where: { status: 'active' }
    });

    // মোট অর্ডার ও রাজস্ব
    const totalOrders = await this.prisma.order.count();
    const revenueData = await this.prisma.order.aggregate({
      _sum: { total: true }
    });

    const todayOrders = await this.prisma.order.findMany({
      where: { createdAt: { gte: today } }
    });

    const todayRevenue = todayOrders.reduce((sum, o) => sum + o.total, 0);

    // মাসিক রাজস্ব
    const monthStart = new Date();
    monthStart.setDate(1);
    monthStart.setHours(0, 0, 0, 0);

    const monthlyOrders = await this.prisma.order.findMany({
      where: { createdAt: { gte: monthStart } }
    });

    const monthlyRevenue = monthlyOrders.reduce((sum, o) => sum + o.total, 0);

    // সার্ভার আপটাইম গণনা করুন
    const uptimeMetrics = await this.calculateUptime();

    // সব মেট্রিক্স সংরক্ষণ করুন
    await this.prisma.platformAnalytics.create({
      data: {
        totalTenants,
        activeTenants,
        totalRevenue: BigInt(Math.round(revenueData._sum.total || 0)) as any,
        monthlyRevenue: BigInt(Math.round(monthlyRevenue)) as any,
        totalStores: totalTenants,
        totalOrders,
        systemUptime: uptimeMetrics
      }
    });

    return {
      totalTenants,
      activeTenants,
      totalRevenue: revenueData._sum.total || 0,
      monthlyRevenue,
      todayRevenue,
      totalOrders,
      systemUptime: `${uptimeMetrics.toFixed(2)}%`
    };
  }

  private async calculateUptime(): Promise<number> {
    // গত ২৪ ঘণ্টার আপটাইম গণনা করুন
    const last24h = new Date(Date.now() - 24 * 60 * 60 * 1000);

    const totalChecks = await this.prisma.uptimeMetrics.count({
      where: { recordedAt: { gte: last24h } }
    });

    const healthyChecks = await this.prisma.uptimeMetrics.count({
      where: {
        recordedAt: { gte: last24h },
        uptime: 100
      }
    });

    return totalChecks > 0 ? (healthyChecks / totalChecks) * 100 : 100;
  }

  // ঐতিহাসিক ডেটা পান
  async getHistoricalMetrics(days: number = 30) {
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - days);

    return this.prisma.platformAnalytics.findMany({
      where: { recordedAt: { gte: startDate } },
      orderBy: { recordedAt: 'asc' }
    });
  }
}