import { Injectable, Logger } from '@nestjs/common';
import * as os from 'os';
import * as si from 'systeminformation';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class ServerDashboardService {
  private logger = new Logger('ServerDashboard');

  constructor(private prisma: PrismaService) {}

  // রিয়েল-টাইম সার্ভার মেট্রিক্স পান
  async getServerMetrics() {
    try {
      // CPU তথ্য
      const cpuLoad = os.loadavg();
      const cpuInfo = await si.cpu();
      const cpuUsage = await si.currentLoad();

      // মেমরি তথ্য
      const memInfo = await si.mem();
      const memUsagePercent = (memInfo.used / memInfo.total) * 100;

      // ডিস্ক তথ্য
      const diskInfo = await si.fsSize();
      const diskUsagePercent = (diskInfo[0].used / diskInfo[0].size) * 100;

      // নেটওয়ার্ক তথ্য
      const networkStats = await si.networkStats();

      // প্রক্রিয়া তথ্য
      const processes = await si.processes();

      // সিস্টেম আপটাইম
      const uptime = os.uptime();

      return {
        cpu: {
          cores: cpuInfo.cores,
          model: cpuInfo.brand,
          loadAverage: cpuLoad,
          currentUsage: cpuUsage.currentLoad.toFixed(2) + '%',
          temperature: cpuUsage.sensors?.[0]?.main || 'N/A'
        },
        memory: {
          total: (memInfo.total / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          used: (memInfo.used / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          free: (memInfo.free / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          usagePercent: memUsagePercent.toFixed(2) + '%'
        },
        disk: {
          total: (diskInfo[0].size / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          used: (diskInfo[0].used / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          free: (diskInfo[0].available / 1024 / 1024 / 1024).toFixed(2) + ' GB',
          usagePercent: diskUsagePercent.toFixed(2) + '%'
        },
        network: {
          interfaces: networkStats.length,
          bytesIn: (networkStats[0].rx_bytes / 1024 / 1024).toFixed(2) + ' MB',
          bytesOut: (networkStats[0].tx_bytes / 1024 / 1024).toFixed(2) + ' MB'
        },
        system: {
          uptime: this.formatUptime(uptime),
          uptimeSeconds: uptime,
          platform: os.platform(),
          arch: os.arch(),
          cpuCount: os.cpus().length,
          hostname: os.hostname()
        },
        processes: {
          total: processes.all,
          running: processes.running
        }
      };
    } catch (err) {
      this.logger.error('Failed to get server metrics:', err);
      throw err;
    }
  }

  // ডাটাবেস মেট্রিক্স
  async getDatabaseMetrics() {
    try {
      // সব টেবিলের রো কাউন্ট
      const stats = {
        users: await this.prisma.user.count(),
        tenants: await this.prisma.tenant.count(),
        products: await this.prisma.product.count(),
        orders: await this.prisma.order.count(),
        customers: await this.prisma.customer.count(),
        transactions: await this.prisma.payment.count()
      };

      return stats;
    } catch (err) {
      this.logger.error('Failed to get database metrics:', err);
      throw err;
    }
  }

  // API পারফরম্যান্স মেট্রিক্স
  async getAPIMetrics() {
    const last24h = new Date(Date.now() - 24 * 60 * 60 * 1000);

    const apiLogs = await this.prisma.aPIUsageLog.findMany({
      where: { createdAt: { gte: last24h } }
    });

    const totalRequests = apiLogs.length;
    const averageResponseTime =
      apiLogs.reduce((sum, log) => sum + log.responseTime, 0) / totalRequests || 0;

    const statusCodes: any = {};
    apiLogs.forEach((log) => {
      statusCodes[log.statusCode] = (statusCodes[log.statusCode] || 0) + 1;
    });

    return {
      totalRequests,
      averageResponseTime: averageResponseTime.toFixed(2) + ' ms',
      statusCodes,
      successRate: (((statusCodes['200'] || 0) / totalRequests) * 100).toFixed(2) + '%'
    };
  }

  // সার্ভার স্বাস্থ্য পরীক্ষা
  async getServerHealth() {
    const metrics = await this.getServerMetrics();
    const health = {
      status: 'healthy',
      warnings: [] as string[],
      errors: [] as string[]
    };

    // CPU চেক
    const cpuUsage = parseFloat(metrics.cpu.currentUsage);
    if (cpuUsage > 80) {
      health.warnings.push(`CPU usage high: ${cpuUsage}%`);
    }
    if (cpuUsage > 95) {
      health.errors.push(`CPU critical: ${cpuUsage}%`);
      health.status = 'critical';
    }

    // মেমরি চেক
    const memUsage = parseFloat(metrics.memory.usagePercent);
    if (memUsage > 80) {
      health.warnings.push(`Memory usage high: ${memUsage}%`);
    }
    if (memUsage > 95) {
      health.errors.push(`Memory critical: ${memUsage}%`);
      health.status = 'critical';
    }

    // ডিস্ক চেক
    const diskUsage = parseFloat(metrics.disk.usagePercent);
    if (diskUsage > 80) {
      health.warnings.push(`Disk usage high: ${diskUsage}%`);
    }
    if (diskUsage > 95) {
      health.errors.push(`Disk critical: ${diskUsage}%`);
      health.status = 'critical';
    }

    if (health.errors.length > 0) {
      health.status = 'critical';
    } else if (health.warnings.length > 0) {
      health.status = 'warning';
    }

    return health;
  }

  private formatUptime(seconds: number): string {
    const days = Math.floor(seconds / (24 * 60 * 60));
    const hours = Math.floor((seconds % (24 * 60 * 60)) / (60 * 60));
    const minutes = Math.floor((seconds % (60 * 60)) / 60);

    return `${days}d ${hours}h ${minutes}m`;
  }
}