import {
  Controller,
  Get,
  Put,
  Delete,
  Param,
  Query,
  UseGuards,
  Req,
  Body
} from '@nestjs/common';
import { ApiTags, ApiBearerAuth } from '@nestjs/swagger';
import { SuperAdminService } from './super-admin.service';
import { JwtAuthGuard } from '../auth/guards/jwt-auth.guard';

// Middleware to check if user is SUPER_ADMIN
const IsSuperAdmin = () => {
  return (target: any, propertyKey?: string, descriptor?: PropertyDescriptor) => {
    const originalMethod = descriptor.value;
    descriptor.value = function (req: any, ...args: any[]) {
      if (req.user.role !== 'SUPER_ADMIN') {
        throw new Error('Unauthorized: Super Admin only');
      }
      return originalMethod.apply(this, [req, ...args]);
    };
  };
};

@ApiTags('Super Admin')
@Controller('super-admin')
@UseGuards(JwtAuthGuard)
@ApiBearerAuth()
export class SuperAdminController {
  constructor(private superAdminService: SuperAdminService) {}

  @Get('tenants')
  @IsSuperAdmin()
  async getAllTenants(
    @Req() req: any,
    @Query('skip') skip = 0,
    @Query('take') take = 20,
    @Query('status') status?: string,
    @Query('plan') plan?: string
  ) {
    return await this.superAdminService.getAllTenants(skip, take, { status, plan });
  }

  @Put('tenants/:id/status')
  @IsSuperAdmin()
  async updateTenantStatus(@Param('id') id: string, @Body() body: { status: string }) {
    return await this.superAdminService.updateTenantStatus(id, body.status);
  }

  @Delete('tenants/:id')
  @IsSuperAdmin()
  async deleteTenant(@Param('id') id: string) {
    return await this.superAdminService.deleteTenant(id);
  }

  @Get('analytics/global')
  @IsSuperAdmin()
  async getGlobalAnalytics(@Req() req: any) {
    return await this.superAdminService.getGlobalAnalytics();
  }

  @Get('analytics/tenant/:id')
  @IsSuperAdmin()
  async getTenantAnalytics(@Param('id') id: string) {
    return await this.superAdminService.getTenantAnalytics(id);
  }

  @Get('health')
  @IsSuperAdmin()
  async getSystemHealth(@Req() req: any) {
    return await this.superAdminService.getSystemHealth();
  }

  @Get('logs')
  @IsSuperAdmin()
  async getAuditLogs(@Query('skip') skip = 0, @Query('take') take = 100) {
    return await this.superAdminService.getAuditLogs(skip, take);
  }
}