import { Injectable, BadRequestException, NotFoundException, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class SuperAdminService {
  private logger = new Logger('SuperAdminService');

  constructor(private prisma: PrismaService) {}

  // Get all tenants (paginated)
  async getAllTenants(skip = 0, take = 20, filters?: any) {
    const where: any = {};

    if (filters?.status) where.status = filters.status;
    if (filters?.plan) where.plan = filters.plan;
    if (filters?.country) where.country = filters.country;

    const [tenants, total] = await Promise.all([
      this.prisma.tenant.findMany({
        where,
        skip,
        take,
        include: { settings: true },
        orderBy: { createdAt: 'desc' }
      }),
      this.prisma.tenant.count({ where })
    ]);

    return {
      items: tenants,
      total,
      page: Math.floor(skip / take) + 1,
      limit: take
    };
  }

  // Suspend/activate tenant
  async updateTenantStatus(tenantId: string, status: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId }
    });

    if (!tenant) {
      throw new NotFoundException('Tenant not found');
    }

    const updated = await this.prisma.tenant.update({
      where: { id: tenantId },
      data: { status: status as any }
    });

    this.logger.log(`Tenant ${tenantId} status changed to ${status}`);

    return {
      success: true,
      data: updated
    };
  }

  // Delete tenant
  async deleteTenant(tenantId: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId }
    });

    if (!tenant) {
      throw new NotFoundException('Tenant not found');
    }

    // Delete all related data (cascading in Prisma)
    await this.prisma.tenant.delete({ where: { id: tenantId } });

    this.logger.log(`Tenant ${tenantId} deleted`);

    return { success: true, message: 'Tenant deleted' };
  }

  // Global analytics dashboard
  async getGlobalAnalytics() {
    const [totalTenants, activeTenants, totalOrders, totalRevenue] = await Promise.all([
      this.prisma.tenant.count(),
      this.prisma.tenant.count({ where: { status: 'ACTIVE' } }),
      this.prisma.order.count(),
      this.prisma.order.aggregate({
        _sum: { total: true }
      })
    ]);

    // Get top tenants by revenue
    const topTenants = await this.prisma.order.groupBy({
      by: ['tenantId'],
      _count: { id: true },
      _sum: { total: true },
      orderBy: { _sum: { total: 'desc' } },
      take: 10
    });

    // Get orders by status
    const ordersByStatus = await this.prisma.order.groupBy({
      by: ['status'],
      _count: { id: true }
    });

    return {
      data: {
        totalTenants,
        activeTenants,
        totalOrders,
        totalRevenue: totalRevenue._sum.total || 0,
        topTenants,
        ordersByStatus
      }
    };
  }

  // Get tenant-specific analytics
  async getTenantAnalytics(tenantId: string) {
    const [totalOrders, totalRevenue, totalProducts, activeUsers] = await Promise.all([
      this.prisma.order.count({ where: { tenantId } }),
      this.prisma.order.aggregate({
        where: { tenantId },
        _sum: { total: true }
      }),
      this.prisma.product.count({ where: { tenantId } }),
      this.prisma.user.count({ where: { tenantId } })
    ]);

    // Get recent orders
    const recentOrders = await this.prisma.order.findMany({
      where: { tenantId },
      take: 10,
      orderBy: { createdAt: 'desc' }
    });

    return {
      data: {
        totalOrders,
        totalRevenue: totalRevenue._sum.total || 0,
        totalProducts,
        activeUsers,
        recentOrders
      }
    };
  }

  // System health/monitoring
  async getSystemHealth() {
    // Check database connection
    const dbHealth = await this.prisma.$queryRaw`SELECT 1`;

    return {
      data: {
        status: dbHealth ? 'healthy' : 'degraded',
        database: 'connected',
        timestamp: new Date().toISOString()
      }
    };
  }

  // View system logs
  async getAuditLogs(skip = 0, take = 100) {
    const logs = await this.prisma.auditLog.findMany({
      skip,
      take,
      orderBy: { createdAt: 'desc' }
    });

    const total = await this.prisma.auditLog.count();

    return {
      items: logs,
      total,
      page: Math.floor(skip / take) + 1,
      limit: take
    };
  }
}