import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import * as nodemailer from 'nodemailer';

@Injectable()
export class TenantManagementService {
  private logger = new Logger('TenantManagementService');
  private mailer = nodemailer.createTransport({
    service: 'gmail',
    auth: {
      user: process.env.GMAIL_USER,
      pass: process.env.GMAIL_PASSWORD
    }
  });

  constructor(private prisma: PrismaService) {}

  // নতুন টেন্যান্ট তৈরি করুন
  async createTenant(tenantData: {
    name: string;
    email: string;
    plan: string;
    country?: string;
  }) {
    const tenant = await this.prisma.tenant.create({
      data: {
        name: tenantData.name,
        email: tenantData.email,
        status: 'active'
      }
    });

    // টেন্যান্ট স্ট্যাটাস তৈরি করুন
    await this.prisma.tenantStatus.create({
      data: {
        tenantId: tenant.id,
        status: 'active'
      }
    });

    // স্বাগত ইমেইল পাঠান
    await this.sendWelcomeEmail(tenant.email, tenant.name);

    this.logger.log(`Tenant created: ${tenant.name}`);

    return { success: true, tenantId: tenant.id };
  }

  // টেন্যান্ট সাস্পেন্ড করুন
  async suspendTenant(tenantId: string, reason: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId }
    });

    if (!tenant) throw new Error('Tenant not found');

    // স্ট্যাটাস আপডেট করুন
    await this.prisma.tenantStatus.update({
      where: { tenantId },
      data: {
        status: 'suspended',
        reason,
        suspendedAt: new Date()
      }
    });

    // বিজ্ঞপ্তি পাঠান
    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL,
      to: tenant.email,
      subject: 'Your store has been suspended',
      html: `<p>Your store has been suspended. Reason: ${reason}</p>`
    });

    this.logger.log(`Tenant suspended: ${tenantId} - Reason: ${reason}`);

    return { success: true };
  }

  // টেন্যান্ট পুনরায় সক্রিয় করুন
  async reactivateTenant(tenantId: string) {
    await this.prisma.tenantStatus.update({
      where: { tenantId },
      data: {
        status: 'active',
        suspendedAt: null,
        reason: null
      }
    });

    const tenant = await this.prisma.tenant.findUnique({
      where: { id: tenantId }
    });

    // পুনরায় সক্রিয়করণ বিজ্ঞপ্তি
    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL,
      to: tenant?.email,
      subject: 'Your store has been reactivated',
      html: `<p>Your store is now active again</p>`
    });

    return { success: true };
  }

  // টেন্যান্ট ডিলিট করুন
  async deleteTenant(tenantId: string) {
    // সম্পর্কিত সব ডেটা ডিলিট করুন (Cascade)
    await this.prisma.tenant.delete({
      where: { id: tenantId }
    });

    this.logger.log(`Tenant deleted: ${tenantId}`);

    return { success: true };
  }

  // সব টেন্যান্ট পান
  async getAllTenants(filters?: { status?: string; country?: string }) {
    return this.prisma.tenant.findMany({
      where: {
        ...(filters?.status && { status: filters.status }),
        ...(filters?.country && { countryCode: filters.country })
      },
      include: { tenantStatus: true }
    });
  }

  private async sendWelcomeEmail(email: string, name: string) {
    await this.mailer.sendMail({
      from: process.env.STORE_EMAIL,
      to: email,
      subject: 'Welcome to our platform!',
      html: `<h1>Welcome ${name}!</h1><p>Your store is ready to go live.</p>`
    });
  }
}