import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class AISupportBotService {
  private logger = new Logger('AISupportBotService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // AI চ্যাটবট প্রতিক্রিয়া জেনারেট করুন
  async generateBotResponse(userMessage: string, tenantId: string): Promise<string> {
    try {
      const systemPrompt = `আপনি একটি বন্ধুত্বপূর্ণ অনলাইন স্টোর সাপোর্ট বটআপনি সব সময় সাহায্য করার চেষ্টা করেন।
সাধারণ প্রশ্নগুলির উত্তর দিন যেমন:
- পণ্য তথ্য
- শিপিং এবং ডেলিভারি
- রিটার্ন পলিসি
- পেমেন্ট অপশন
- অ্যাকাউন্ট সমস্যা

যদি আপনি সমস্যার সমাধান করতে না পারেন, তবে লাইভ এজেন্টের কাছে স্থানান্তরের পরামর্শ দিন।`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userMessage }
        ],
        temperature: 0.7,
        max_tokens: 300
      });

      const botResponse = response.choices[0].message.content || '';

      // ইন্টারঅ্যাকশন সংরক্ষণ করুন
      await this.prisma.aIBotInteraction.create({
        data: {
          tenantId,
          userMessage,
          botResponse,
          confidence: 0.85
        }
      });

      return botResponse;
    } catch (err) {
      this.logger.error('AI response generation failed:', err);
      return 'I apologize, but I encountered an issue. Please contact our support team for assistance.';
    }
  }

  // ২৪/৭ উপলব্ধতা
  async handleBotMessage(
    userId: string | null,
    sessionId: string,
    message: string,
    tenantId: string
  ) {
    // AI প্রতিক্রিয়া জেনারেট করুন
    const response = await this.generateBotResponse(message, tenantId);

    // যদি সমস্যা সমাধান করা যায় না, তবে লাইভ এজেন্টকে এসকেলেট করুন
    if (response.toLowerCase().includes('support team')) {
      // লাইভ চ্যাটে স্থানান্তর করুন
      return {
        response,
        escalateToAgent: true
      };
    }

    return { response, escalateToAgent: false };
  }
}