import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';
import axios from 'axios';

@Injectable()
export class CommunicationsService {
  private logger = new Logger('CommunicationsService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // ===== IN-APP ANNOUNCEMENTS =====
  async createAnnouncement(
    tenantId: string,
    announcementData: {
      title: string;
      content: string;
      type: 'feature' | 'maintenance' | 'update' | 'alert';
      actionUrl?: string;
      expiresAt?: Date;
    }
  ) {
    const announcement = await this.prisma.inAppAnnouncement.create({
      data: {
        tenantId,
        ...announcementData,
        isActive: true
      }
    });

    return { success: true, announcementId: announcement.id };
  }

  async getActiveAnnouncements(tenantId: string) {
    const now = new Date();

    return this.prisma.inAppAnnouncement.findMany({
      where: {
        tenantId,
        isActive: true,
        OR: [
          { expiresAt: null },
          { expiresAt: { gt: now } }
        ]
      },
      orderBy: { createdAt: 'desc' }
    });
  }

  // ===== WHATSAPP/TELEGRAM INTEGRATION =====
  async configureMessagingPlatform(
    tenantId: string,
    platform: 'whatsapp' | 'telegram',
    credentials: any
  ) {
    const integration = await this.prisma.messagingIntegration.upsert({
      where: { tenantId },
      create: {
        tenantId,
        platform,
        ...credentials,
        isActive: true
      },
      update: {
        ...credentials,
        isActive: true
      }
    });

    return { success: true, data: integration };
  }

  async sendViaWhatsApp(phoneNumber: string, message: string) {
    // WhatsApp API integration (Twilio, Meta, etc.)
    this.logger.log(`WhatsApp message sent to ${phoneNumber}`);
    return { success: true };
  }

  async sendViaTelegram(chatId: string, message: string) {
    try {
      const botToken = process.env.TELEGRAM_BOT_TOKEN;
      
      await axios.post(
        `https://api.telegram.org/bot${botToken}/sendMessage`,
        {
          chat_id: chatId,
          text: message
        }
      );

      this.logger.log(`Telegram message sent to ${chatId}`);
      return { success: true };
    } catch (err) {
      this.logger.error('Telegram send failed:', err);
      return { success: false };
    }
  }

  // ===== SUPPORT RATING & FEEDBACK =====
  async rateSupport(
    conversationId: string,
    userId: string,
    rating: number,
    feedback?: string
  ) {
    const supportRating = await this.prisma.supportRating.create({
      data: {
        conversationId,
        userId,
        rating,
        feedback
      }
    });

    return { success: true, ratingId: supportRating.id };
  }

  async getSupportRatings(tenantId: string) {
    // টেন্যান্টের সব রেটিং পান
    const ratings = await this.prisma.supportRating.findMany();

    const avgRating = ratings.length > 0
      ? ratings.reduce((sum, r) => sum + r.rating, 0) / ratings.length
      : 0;

    return {
      averageRating: avgRating,
      totalRatings: ratings.length,
      feedbackList: ratings.map(r => r.feedback).filter(Boolean)
    };
  }

  // ===== AI ASSISTANT =====
  async askAIAssistant(
    tenantId: string,
    userId: string,
    question: string
  ): Promise<string> {
    try {
      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [
          {
            role: 'system',
            content: `You are a helpful e-commerce platform assistant. Help store owners with:
            - Setting up their store
            - Configuring products
            - Payment integration
            - Shipping setup
            - Troubleshooting common issues
            Answer in the user's language.`
          },
          {
            role: 'user',
            content: question
          }
        ],
        temperature: 0.7,
        max_tokens: 500
      });

      const answer = response.choices[0].message.content || '';

      // কথোপকথন সংরক্ষণ করুন
      await this.prisma.aIAssistantChat.create({
        data: {
          tenantId,
          userId,
          question,
          answer,
          category: this.categorizeQuestion(question)
        }
      });

      return answer;
    } catch (err) {
      this.logger.error('AI Assistant failed:', err);
      throw err;
    }
  }

  private categorizeQuestion(question: string): string {
    const lowerQ = question.toLowerCase();

    if (lowerQ.includes('setup') || lowerQ.includes('configure')) {
      return 'setup';
    } else if (lowerQ.includes('payment') || lowerQ.includes('gateway')) {
      return 'integration';
    } else if (lowerQ.includes('error') || lowerQ.includes('problem')) {
      return 'troubleshooting';
    }

    return 'general';
  }
}