import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class LiveChatService {
  private logger = new Logger('LiveChatService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // চ্যাট কথোপকথন শুরু করুন
  async startConversation(tenantId: string, userId: string) {
    const conversation = await this.prisma.chatConversation.create({
      data: {
        tenantId,
        userId,
        status: 'open'
      }
    });

    // AI বট দিয়ে স্বাগত বার্তা পাঠান
    const welcomeMessage = await this.generateWelcomeMessage();

    await this.prisma.chatMessage.create({
      data: {
        conversationId: conversation.id,
        senderType: 'bot',
        senderId: 'ai-bot',
        message: welcomeMessage
      }
    });

    return {
      conversationId: conversation.id,
      welcomeMessage
    };
  }

  // ব্যবহারকারী বার্তা পাঠান
  async sendMessage(conversationId: string, userId: string, message: string) {
    // ব্যবহারকারী বার্তা সংরক্ষণ করুন
    await this.prisma.chatMessage.create({
      data: {
        conversationId,
        senderType: 'user',
        senderId: userId,
        message
      }
    });

    // AI বট দিয়ে উত্তর তৈরি করুন
    const botResponse = await this.generateBotResponse(message);

    // Bot প্রতিক্রিয়া সংরক্ষণ করুন
    const botMessage = await this.prisma.chatMessage.create({
      data: {
        conversationId,
        senderType: 'bot',
        senderId: 'ai-bot',
        message: botResponse
      }
    });

    // যদি বট সমাধান করতে না পারে, সাপোর্ট এজেন্টকে এসকেলেট করুন
    if (this.shouldEscalate(botResponse)) {
      await this.prisma.chatConversation.update({
        where: { id: conversationId },
        data: {
          status: 'waiting',
          assignedAgent: null // এজেন্ট অ্যাসাইন করার জন্য অপেক্ষা করুন
        }
      });
    }

    return { success: true, botMessage };
  }

  // AI বট প্রতিক্রিয়া তৈরি করুন
  private async generateBotResponse(userMessage: string): Promise<string> {
    try {
      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [
          {
            role: 'system',
            content: `You are a helpful customer support assistant. Answer questions about e-commerce, 
            product setup, payments, and general inquiries. If you cannot help, suggest escalating to a human agent.`
          },
          {
            role: 'user',
            content: userMessage
          }
        ],
        temperature: 0.7,
        max_tokens: 300
      });

      return response.choices[0].message.content || 'আমি সাহায্য করতে পারছি না। অনুগ্রহ করে একজন এজেন্টের সাথে কথা বলুন।';
    } catch (err) {
      this.logger.error('Bot response generation failed:', err);
      return 'দুঃখিত, একটি ত্রুটি ঘটেছে। অনুগ্রহ করে পুনরায় চেষ্টা করুন।';
    }
  }

  private shouldEscalate(response: string): boolean {
    // যদি বট "এজেন্ট" বা "স্কেলেট" শব্দ ব্যবহার করে
    return response.toLowerCase().includes('agent') ||
           response.toLowerCase().includes('escalate') ||
           response.toLowerCase().includes('সমর্থন');
  }

  private async generateWelcomeMessage(): Promise<string> {
    return `👋 স্বাগতম! আমি আপনাকে সাহায্য করতে এখানে আছি। আপনার প্রশ্ন কী?`;
  }

  // সব চ্যাট কথোপকথন পান
  async getConversations(tenantId: string) {
    return this.prisma.chatConversation.findMany({
      where: { tenantId },
      include: { messages: true },
      orderBy: { startedAt: 'desc' }
    });
  }
}