import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { InjectQueue } from '@nestjs/bull';
import { Queue } from 'bull';

@Injectable()
export class AutoPriceSyncService {
  private logger = new Logger('AutoPriceSyncService');

  constructor(
    private prisma: PrismaService,
    @InjectQueue('price-sync') private priceSyncQueue: Queue
  ) {}

  // প্রাইস পরিবর্তন ট্রিগার করুন
  async triggerPriceUpdate(
    wholesalerTenantId: string,
    productId: string,
    newPrice: number
  ) {
    // কিউতে যোগ করুন
    await this.priceSyncQueue.add('sync-price', {
      wholesalerTenantId,
      productId,
      newPrice
    });

    return { success: true };
  }

  // প্রাইস সিঙ্ক প্রসেসর
  async processPriceSync(wholesalerTenantId: string, productId: string, newPrice: number) {
    // সাবস্ক্রাইবড স্টোরগুলিতে প্রাইস আপডেট করুন
    const connections = await this.prisma.adminWholesalerConnection.findMany({
      where: {
        wholesalerTenantId,
        autoSyncPrices: true
      }
    });

    for (const connection of connections) {
      const adminProduct = await this.prisma.product.findFirst({
        where: {
          tenantId: connection.adminTenantId,
          sku: (
            await this.prisma.product.findUnique({
              where: { id: productId },
              select: { sku: true }
            })
          )?.sku
        }
      });

      if (adminProduct) {
        await this.prisma.product.update({
          where: { id: adminProduct.id },
          data: { price: newPrice }
        });
      }
    }

    this.logger.log(`Price synced for product ${productId}`);
  }
}