import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class ProductSyncService {
  private logger = new Logger('ProductSyncService');

  constructor(private prisma: PrismaService) {}

  // নতুন কানেকশন রেজিস্টার করুন
  async registerConnection(adminTenantId: string, wholesalerTenantId: string) {
    const connection = await this.prisma.adminWholesalerConnection.create({
      data: {
        adminTenantId,
        wholesalerTenantId,
        status: 'active'
      }
    });

    return { success: true, data: connection };
  }

  // প্রতি 5 মিনিটে স্টক সিঙ্ক করুন
  @Cron('*/5 * * * *')
  async syncStockAndPrices() {
    const connections = await this.prisma.adminWholesalerConnection.findMany({
      where: { status: 'active' }
    });

    for (const connection of connections) {
      try {
        // হোলসেলার প্রোডাক্ট পান
        const wholesalerProducts = await this.prisma.product.findMany({
          where: { tenantId: connection.wholesalerTenantId }
        });

        for (const product of wholesalerProducts) {
          // অ্যাডমিন স্টোরে প্রোডাক্ট খুঁজুন
          const adminProduct = await this.prisma.product.findFirst({
            where: {
              tenantId: connection.adminTenantId,
              sku: product.sku
            }
          });

          if (!adminProduct) continue;

          // স্টক আপডেট করুন
          if (connection.autoSyncStock) {
            await this.prisma.product.update({
              where: { id: adminProduct.id },
              data: { stock: product.stock }
            });
          }

          // প্রাইস আপডেট করুন
          if (connection.autoSyncPrices) {
            await this.prisma.product.update({
              where: { id: adminProduct.id },
              data: { price: product.price }
            });

            // প্রাইস চেঞ্জ নোটিফিকেশন পাঠান
            if (adminProduct.price !== product.price) {
              await this.notifyPriceChange(adminProduct.id, adminProduct.price, product.price);
            }
          }
        }

        this.logger.log(`Sync completed: ${connection.adminTenantId}`);
      } catch (err) {
        this.logger.warn(`Sync failed for connection ${connection.id}`, err);
      }
    }
  }

  private async notifyPriceChange(productId: string, oldPrice: number, newPrice: number) {
    // নোটিফিকেশন পাঠান
    // TODO: Send email/SMS notification
  }
}