import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Decimal } from '@prisma/client/runtime/library';

@Injectable()
export class CountryTaxService {
  private logger = new Logger('CountryTaxService');

  constructor(private prisma: PrismaService) {}

  // দেশের ট্যাক্স কনফিগার করুন
  async configureCountryTax(
    countryCode: string,
    taxType: string,
    taxRate: number,
    productCategory?: string
  ) {
    const taxConfig = await this.prisma.countryTaxConfig.upsert({
      where: {
        countryCode_taxType_productCategory: {
          countryCode,
          taxType,
          productCategory: productCategory || null
        }
      } as any,
      create: {
        countryCode,
        taxType,
        taxRate,
        productCategory
      },
      update: { taxRate }
    });

    this.logger.log(`Tax configured: ${taxType} ${taxRate}% in ${countryCode}`);

    return { success: true, data: taxConfig };
  }

  // ট্যাক্স ক্যালকুলেট করুন
  async calculateTax(
    countryCode: string,
    amount: number,
    productCategory?: string
  ): Promise<number> {
    // পণ্য ক্যাটেগরি-নির্দিষ্ট ট্যাক্স খুঁজুন
    let taxConfig = await this.prisma.countryTaxConfig.findFirst({
      where: {
        countryCode,
        productCategory
      }
    });

    // যদি না পাওয়া যায়, সাধারণ ট্যাক্স খুঁজুন
    if (!taxConfig) {
      taxConfig = await this.prisma.countryTaxConfig.findFirst({
        where: {
          countryCode,
          productCategory: null
        }
      });
    }

    if (!taxConfig) {
      return 0; // কোনো ট্যাক্স নেই
    }

    const taxAmount = amount * (taxConfig.taxRate / 100);
    return Math.round(taxAmount * 100) / 100; // 2 ডেসিমাল প্লেসে রাউন্ড করুন
  }

  // ইনভয়েস টেক্স ইনফরমেশন
  async getInvoiceTaxInfo(
    countryCode: string,
    orderItems: any[]
  ) {
    let subtotal = 0;
    let totalTax = 0;

    for (const item of orderItems) {
      subtotal += item.price * item.quantity;

      const tax = await this.calculateTax(
        countryCode,
        item.price * item.quantity,
        item.product?.categoryId
      );

      totalTax += tax;
    }

    return {
      subtotal,
      tax: totalTax,
      total: subtotal + totalTax,
      taxRate: (totalTax / subtotal) * 100
    };
  }

  // কমপ্লায়েন্স রিপোর্ট (সরকারের জন্য)
  async generateComplianceReport(countryCode: string, month: number, year: number) {
    // মাসের অর্ডার এবং ট্যাক্স ডেটা সংগ্রহ করুন
    const startDate = new Date(year, month - 1, 1);
    const endDate = new Date(year, month, 0);

    // TODO: Generate tax report for government compliance

    return {
      country: countryCode,
      month,
      year,
      totalRevenue: 0,
      totalTax: 0,
      orders: 0
    };
  }
}