import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class TeamService {
  private logger = new Logger('TeamService');

  constructor(private prisma: PrismaService) {}

  // টিম মেম্বার যোগ করুন
  async addTeamMember(
    tenantId: string,
    email: string,
    role: 'owner' | 'admin' | 'manager' | 'staff' | 'viewer'
  ) {
    // ইউজার খুঁজুন বা তৈরি করুন
    let user = await this.prisma.user.findUnique({ where: { email } });

    if (!user) {
      // Invite email পাঠান
      user = await this.prisma.user.create({
        data: {
          email,
          name: email.split('@')[0],
          status: 'invited'
        }
      });
    }

    // টিম মেম্বার যোগ করুন
    const member = await this.prisma.teamMember.create({
      data: {
        tenantId,
        userId: user.id,
        role,
        permissions: this.getDefaultPermissions(role)
      }
    });

    this.logger.log(`Team member added: ${email} as ${role}`);

    // Invite email পাঠান
    // await this.sendInviteEmail(email, tenantId);

    return { success: true, data: member };
  }

  // অনুমতি সেট করুন
  private getDefaultPermissions(role: string): any {
    const permissionsMap: any = {
      owner: {
        all: true // সব কিছু এক্সেস
      },
      admin: {
        products: ['create', 'read', 'update', 'delete'],
        orders: ['read', 'update'],
        customers: ['read'],
        team: ['read', 'create'],
        settings: ['read', 'update']
      },
      manager: {
        products: ['read', 'update'],
        orders: ['read', 'update'],
        customers: ['read'],
        analytics: ['read']
      },
      staff: {
        products: ['read'],
        orders: ['read', 'update'],
        customers: ['read']
      },
      viewer: {
        products: ['read'],
        orders: ['read'],
        analytics: ['read']
      }
    };

    return permissionsMap[role] || {};
  }

  // টিম মেম্বার রোল আপডেট করুন
  async updateMemberRole(tenantId: string, userId: string, newRole: string) {
    const member = await this.prisma.teamMember.update({
      where: { tenantId_userId: { tenantId, userId } },
      data: {
        role: newRole,
        permissions: this.getDefaultPermissions(newRole)
      }
    });

    return { success: true, data: member };
  }

  // টিম মেম্বার রিমুভ করুন
  async removeMember(tenantId: string, userId: string) {
    await this.prisma.teamMember.delete({
      where: { tenantId_userId: { tenantId, userId } }
    });

    return { success: true };
  }

  // টিম লিস্ট পান
  async getTeam(tenantId: string) {
    const members = await this.prisma.teamMember.findMany({
      where: { tenantId },
      include: { user: { select: { name: true, email: true } } }
    });

    return { success: true, data: members };
  }
}