import { Injectable, BadRequestException, ConflictException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { CreateTenantDto } from './dto/create-tenant.dto';
import { normalizeSubdomain, validateSubdomain } from '@saas/utils';

@Injectable()
export class TenantService {
  constructor(private prisma: PrismaService) {}

  async createTenant(dto: CreateTenantDto) {
    const normalized = normalizeSubdomain(dto.subdomain);

    if (!validateSubdomain(normalized)) {
      throw new BadRequestException('Invalid subdomain format');
    }

    // Check uniqueness
    const existing = await this.prisma.tenant.findUnique({
      where: { subdomain: normalized }
    });

    if (existing) {
      throw new ConflictException('Subdomain already taken');
    }

    const baseHost = process.env.NEXT_PUBLIC_BASE_HOST || 'example.com';
    const host = `${normalized}.${baseHost}`;

    const tenant = await this.prisma.tenant.create({
      data: {
        name: dto.name,
        subdomain: normalized,
        host,
        country: dto.country || 'US',
        currency: dto.currency || 'USD',
        status: 'ACTIVE',
        settings: {
          create: {
            storeName: dto.name,
            storeEmail: dto.email || 'noreply@' + host
          }
        }
      },
      include: { settings: true }
    });

    return {
      success: true,
      data: tenant
    };
  }

  async findTenantById(id: string) {
    const tenant = await this.prisma.tenant.findUnique({
      where: { id },
      include: { settings: true }
    });

    if (!tenant) {
      throw new BadRequestException('Tenant not found');
    }

    return tenant;
  }

  async findAllTenants(skip = 0, take = 10) {
    const [tenants, total] = await Promise.all([
      this.prisma.tenant.findMany({ skip, take }),
      this.prisma.tenant.count()
    ]);

    return {
      items: tenants,
      total,
      page: Math.floor(skip / take) + 1,
      limit: take
    };
  }
}