import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { PREMIUM_THEMES } from './all-themes';

@Injectable()
export class ThemeService {
  private logger = new Logger('ThemeService');

  constructor(private prisma: PrismaService) {
    this.initializeThemes();
  }

  // সব থিম ইনিশিয়ালাইজ করুন
  async initializeThemes() {
    for (const theme of PREMIUM_THEMES) {
      const existing = await this.prisma.premiumTheme.findUnique({
        where: { themeId: theme.id }
      });

      if (!existing) {
        await this.prisma.premiumTheme.create({
          data: {
            themeId: theme.id,
            name: theme.name,
            description: theme.description,
            category: theme.category,
            colors: theme.colors,
            fonts: theme.fonts,
            components: theme.components,
            isActive: true
          }
        });

        this.logger.log(`Theme initialized: ${theme.name}`);
      }
    }
  }

  // সব থিম পান
  async getAllThemes() {
    return PREMIUM_THEMES;
  }

  // ক্যাটেগরি অনুযায়ী থিম পান
  async getThemesByCategory(category: string) {
    return PREMIUM_THEMES.filter((theme) => theme.category === category);
  }

  // থিম প্রয়োগ করুন
  async applyTheme(tenantId: string, themeId: string, customizations?: any) {
    const theme = PREMIUM_THEMES.find((t) => t.id === themeId);

    if (!theme) {
      throw new Error('Theme not found');
    }

    const usage = await this.prisma.themeUsage.upsert({
      where: { tenantId_themeId: { tenantId, themeId } },
      create: {
        tenantId,
        themeId,
        customizations: customizations || {}
      },
      update: {
        customizations: customizations || {}
      }
    });

    this.logger.log(`Theme applied: ${theme.name} to store ${tenantId}`);

    return {
      success: true,
      data: {
        theme,
        customizations: usage.customizations
      }
    };
  }

  // কাস্টম থিম তৈরি করুন
  async createCustomTheme(tenantId: string, customTheme: any) {
    const baseTheme = PREMIUM_THEMES[0]; // ডিফল্ট থিম হিসেবে ব্যবহার

    const customization = {
      ...baseTheme,
      ...customTheme,
      id: `custom-${tenantId}-${Date.now()}`
    };

    const usage = await this.prisma.themeUsage.create({
      data: {
        tenantId,
        themeId: customization.id,
        customizations: customization
      }
    });

    return {
      success: true,
      data: customization
    };
  }

  // কারেন্ট থিম পান
  async getCurrentTheme(tenantId: string) {
    const usage = await this.prisma.themeUsage.findFirst({
      where: { tenantId },
      orderBy: { appliedAt: 'desc' }
    });

    if (!usage) {
      return { theme: PREMIUM_THEMES[0] };
    }

    const theme = PREMIUM_THEMES.find((t) => t.id === usage.themeId);

    return {
      theme: {
        ...theme,
        ...usage.customizations
      }
    };
  }
}