import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class VersionControlService {
  constructor(private prisma: PrismaService) {}

  // থিম স্ন্যাপশট তৈরি করুন
  async createThemeSnapshot(
    tenantId: string,
    themeConfig: any,
    reason: string = 'manual'
  ) {
    // সর্বোচ্চ ভার্সন খুঁজুন
    const latest = await this.prisma.themeBackup.findFirst({
      where: { tenantId },
      orderBy: { version: 'desc' }
    });

    const newVersion = (latest?.version || 0) + 1;

    const backup = await this.prisma.themeBackup.create({
      data: {
        tenantId,
        version: newVersion,
        themeName: themeConfig.name,
        themeConfig,
        backupReason: reason
      }
    });

    return {
      success: true,
      version: newVersion,
      backupId: backup.id
    };
  }

  // ভার্সন হিস্টরি পান
  async getVersionHistory(tenantId: string) {
    return this.prisma.themeBackup.findMany({
      where: { tenantId },
      orderBy: { version: 'desc' }
    });
  }

  // নির্দিষ্ট ভার্সন পান
  async getVersion(backupId: string) {
    return this.prisma.themeBackup.findUnique({
      where: { id: backupId }
    });
  }

  // রোলব্যাক করুন
  async rollbackToVersion(tenantId: string, version: number) {
    const backup = await this.prisma.themeBackup.findFirst({
      where: { tenantId, version }
    });

    if (!backup) throw new Error('Version not found');

    // রোলব্যাক করুন (থিম আপডেট করুন)
    // TODO: Apply theme configuration

    return {
      success: true,
      message: `Rolled back to version ${version}`
    };
  }

  // ভার্সন তুলনা করুন
  async compareVersions(backupId1: string, backupId2: string) {
    const backup1 = await this.prisma.themeBackup.findUnique({
      where: { id: backupId1 }
    });

    const backup2 = await this.prisma.themeBackup.findUnique({
      where: { id: backupId2 }
    });

    if (!backup1 || !backup2) throw new Error('Backup not found');

    // পার্থক্য খুঁজুন
    const diff = {
      v1: backup1.version,
      v2: backup2.version,
      changes: this.getConfigDifferences(
        backup1.themeConfig,
        backup2.themeConfig
      )
    };

    return diff;
  }

  private getConfigDifferences(config1: any, config2: any): any[] {
    const changes: any[] = [];

    // রিকার্সিভ তুলনা
    const compare = (obj1: any, obj2: any, path = '') => {
      Object.keys({ ...obj1, ...obj2 }).forEach((key) => {
        const val1 = obj1?.[key];
        const val2 = obj2?.[key];
        const currentPath = path ? `${path}.${key}` : key;

        if (typeof val1 === 'object' && typeof val2 === 'object') {
          compare(val1, val2, currentPath);
        } else if (val1 !== val2) {
          changes.push({
            field: currentPath,
            from: val1,
            to: val2
          });
        }
      });
    };

    compare(config1, config2);
    return changes;
  }
}