import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import axios from 'axios';

@Injectable()
export class WebhookEventsService {
  private logger = new Logger('WebhookEvents');

  constructor(private prisma: PrismaService) {}

  // সব সম্ভাব্য ইভেন্ট রেজিস্টার করুন
  async registerWebhookEvent(
    tenantId: string,
    event: string,
    webhookUrl: string
  ) {
    const webhook = await this.prisma.webhookEndpoint.create({
      data: {
        tenantId,
        url: webhookUrl,
        events: [event],
        isActive: true
      }
    });

    return { success: true, webhookId: webhook.id };
  }

  // সমস্ত ইভেন্ট ট্রিগার করুন
  async triggerEvent(tenantId: string, event: string, payload: any) {
    const webhooks = await this.prisma.webhookEndpoint.findMany({
      where: {
        tenantId,
        events: { has: event },
        isActive: true
      }
    });

    const eventLog = {
      event,
      tenantId,
      timestamp: new Date(),
      payload
    };

    // সব registered webhooks-এ পাঠান
    for (const webhook of webhooks) {
      try {
        await axios.post(webhook.url, eventLog, {
          timeout: 5000,
          headers: {
            'X-Webhook-Event': event,
            'X-Webhook-Timestamp': new Date().toISOString()
          }
        });

        this.logger.log(`Event ${event} sent to ${webhook.url}`);
      } catch (err) {
        this.logger.error(`Failed to send webhook ${webhook.id}:`, err);
        // রিট্রাই লজিক
      }
    }
  }

  // Available Events List:
  getAvailableEvents() {
    return [
      'product.created',
      'product.updated',
      'product.deleted',
      'order.created',
      'order.updated',
      'order.shipped',
      'order.delivered',
      'payment.received',
      'payment.failed',
      'customer.created',
      'customer.updated',
      'inventory.low',
      'inventory.out_of_stock',
      'review.created',
      'review.approved'
    ];
  }
}