import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class CountryWholesalerService {
  private logger = new Logger('CountryWholesalerService');

  constructor(private prisma: PrismaService) {}

  // দেশ-ভিত্তিক হোলসেলার নেটওয়ার্ক
  async registerCountryWholesaler(
    countryCode: string,
    wholesalerData: {
      companyName: string;
      contactEmail: string;
      contactPhone: string;
      address: string;
      city: string;
      bankDetails?: any;
    }
  ) {
    // দেশ যাচাই করুন
    const country = await this.prisma.countryConfig.findUnique({
      where: { countryCode }
    });

    if (!country || !country.isActive) {
      throw new Error(`Country ${countryCode} is not active`);
    }

    const wholesaler = await this.prisma.countryWholesaler.create({
      data: {
        countryCode,
        ...wholesalerData
      }
    });

    this.logger.log(`Wholesaler registered in ${countryCode}: ${wholesaler.companyName}`);

    return {
      success: true,
      wholesalerId: wholesaler.id,
      countryCode
    };
  }

  // দেশের সব হোলসেলার পান
  async getCountryWholesalers(countryCode: string) {
    return this.prisma.countryWholesaler.findMany({
      where: {
        countryCode,
        isVerified: true
      }
    });
  }

  // হোলসেলার পোর্টাল
  async getWholesalerDashboard(wholesalerId: string) {
    const wholesaler = await this.prisma.countryWholesaler.findUnique({
      where: { id: wholesalerId }
    });

    if (!wholesaler) throw new Error('Wholesaler not found');

    // দেশের স্টোর পান
    const stores = await this.prisma.tenant.findMany({
      where: { countryCode: wholesaler.countryCode }
    });

    // সেলস ডেটা
    const salesData = await this.getSalesByWholesaler(
      wholesalerId,
      wholesaler.countryCode
    );

    return {
      wholesaler,
      stores: stores.length,
      salesData,
      dashboard: {
        activeOrders: 0,
        pendingPayouts: 0,
        monthlyRevenue: 0
      }
    };
  }

  private async getSalesByWholesaler(
    wholesalerId: string,
    countryCode: string
  ) {
    // Wholesaler-এর সেলস ট্র্যাকিং
    // TODO: Implement tracking logic
    return {};
  }

  // হোলসেলার যাচাই করুন
  async verifyWholesaler(wholesalerId: string) {
    const wholesaler = await this.prisma.countryWholesaler.update({
      where: { id: wholesalerId },
      data: { isVerified: true }
    });

    this.logger.log(`Wholesaler verified: ${wholesaler.companyName}`);

    return { success: true };
  }
}