import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import OpenAI from 'openai';

@Injectable()
export class DemandPredictorService {
  private logger = new Logger('DemandPredictorService');
  private openai: OpenAI;

  constructor(private prisma: PrismaService) {
    this.openai = new OpenAI({ apiKey: process.env.OPENAI_API_KEY });
  }

  // পণ্য ডিমান্ড প্রেডিক্ট করুন
  async predictDemand(
    wholesalerId: string,
    productId: string,
    forecastDays: number = 30
  ) {
    try {
      // হিস্টোরিকাল অর্ডার ডেটা পান
      const orders = await this.prisma.wholesaleOrder.findMany({
        where: { wholesalerId }
      });

      // পণ্য অর্ডার ডেটা এক্সট্র্যাক্ট করুন
      const productOrders = [];
      for (const order of orders) {
        const items = order.items as any[];
        const item = items.find((i) => i.productId === productId);
        if (item) {
          productOrders.push({
            date: order.createdAt,
            quantity: item.qty
          });
        }
      }

      if (productOrders.length < 5) {
        return {
          prediction: 'Insufficient data',
          confidence: 0
        };
      }

      // AI দিয়ে ট্রেন্ড বিশ্লেষণ করুন
      const prompt = `
Analyze this product order history and predict demand for next ${forecastDays} days:
${JSON.stringify(productOrders.slice(-30))}

Provide:
1. Predicted daily demand quantity
2. Confidence level (0-1)
3. Seasonal patterns
4. Growth trend

Return JSON format only:
{
  "predictedDailyQty": number,
  "confidence": number,
  "trend": "increasing|stable|decreasing",
  "recommendation": "stock more|maintain|reduce"
}
`;

      const response = await this.openai.chat.completions.create({
        model: 'gpt-3.5-turbo',
        messages: [{ role: 'user', content: prompt }],
        temperature: 0.5,
        max_tokens: 300
      });

      const text = response.choices[0].message.content || '';
      const jsonMatch = text.match(/\{[\s\S]*\}/);
      const prediction = jsonMatch ? JSON.parse(jsonMatch[0]) : {};

      // ডেমান্ড ফোরকাস্ট সংরক্ষণ করুন
      await this.prisma.demandForecast.create({
        data: {
          wholesalerId,
          productId,
          predictedQty: prediction.predictedDailyQty || 0,
          confidence: prediction.confidence || 0,
          forecastDate: new Date(Date.now() + forecastDays * 24 * 60 * 60 * 1000)
        }
      });

      return {
        success: true,
        data: prediction
      };
    } catch (err) {
      this.logger.error('Demand prediction failed:', err);
      throw err;
    }
  }
}