import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';

@Injectable()
export class DistributionService {
  private logger = new Logger('DistributionService');

  constructor(private prisma: PrismaService) {}

  // হোলসেলার প্রোডাক্ট সাবস্ক্রাইব করুন
  async subscribeToProduct(
    wholesalerId: string,
    tenantId: string,
    productId: string,
    markupPercent: number = 0
  ) {
    const subscription = await this.prisma.wholesaleSubscriber.create({
      data: {
        wholesalerId,
        tenantId,
        productId,
        markupPercent,
        syncPrice: true,
        syncStock: true,
        syncImages: true
      }
    });

    // প্রোডাক্ট স্টোরে যোগ করুন
    await this.addProductToStore(tenantId, productId, wholesalerId, markupPercent);

    this.logger.log(`Product subscribed: ${tenantId} <- ${wholesalerId}`);

    return { success: true, data: subscription };
  }

  // রিয়েল-টাইম প্রাইস/স্টক সিঙ্ক (প্রতি 5 মিনিটে)
  @Cron('*/5 * * * *')
  async syncProductData() {
    const subscriptions = await this.prisma.wholesaleSubscriber.findMany({
      include: { product: true, wholesaler: true }
    });

    for (const sub of subscriptions) {
      try {
        if (sub.syncPrice) {
          // প্রাইস আপডেট করুন
          const newPrice = this.calculatePriceWithMarkup(
            sub.product.basePrice,
            sub.markupPercent
          );

          await this.prisma.product.updateMany({
            where: {
              tenantId: sub.tenantId,
              sku: sub.product.sku
            },
            data: { price: newPrice }
          });
        }

        if (sub.syncStock) {
          // স্টক আপডেট করুন
          await this.prisma.product.updateMany({
            where: {
              tenantId: sub.tenantId,
              sku: sub.product.sku
            },
            data: { stock: sub.product.stock }
          });
        }

        if (sub.syncImages) {
          // ইমেজ আপডেট করুন
          await this.prisma.product.updateMany({
            where: {
              tenantId: sub.tenantId,
              sku: sub.product.sku
            },
            data: { images: sub.product.images }
          });
        }

        // সিঙ্ক টাইম আপডেট করুন
        await this.prisma.wholesaleSubscriber.update({
          where: { id: sub.id },
          data: { lastSyncedAt: new Date() }
        });
      } catch (err) {
        this.logger.warn(`Sync failed for subscription ${sub.id}`, err);
      }
    }
  }

  private calculatePriceWithMarkup(basePrice: number, markupPercent: number): number {
    return basePrice * (1 + markupPercent / 100);
  }

  private async addProductToStore(
    tenantId: string,
    productId: string,
    wholesalerId: string,
    markupPercent: number
  ) {
    const wholesaleProduct = await this.prisma.wholesaleProduct.findUnique({
      where: { id: productId }
    });

    if (!wholesaleProduct) return;

    const storePrice = this.calculatePriceWithMarkup(
      Number(wholesaleProduct.basePrice),
      markupPercent
    );

    // স্টোরে প্রোডাক্ট যোগ করুন
    await this.prisma.product.create({
      data: {
        tenantId,
        sku: wholesaleProduct.sku,
        title: wholesaleProduct.title,
        description: wholesaleProduct.description,
        price: storePrice,
        stock: wholesaleProduct.stock,
        images: wholesaleProduct.images
      }
    });
  }
}