import { Injectable, Logger } from '@nestjs/common';
import * as PDFDocument from 'pdfkit';
import * as AWS from 'aws-sdk';

@Injectable()
export class DocumentGeneratorService {
  private logger = new Logger('DocumentGeneratorService');
  private s3 = new AWS.S3();

  // শিপিং লেবেল জেনারেট করুন
  async generateShippingLabel(order: any): Promise<string> {
    const doc = new PDFDocument();
    const fileName = `shipping-label-${order.id}.pdf`;

    // লেবেল কন্টেন্ট তৈরি করুন
    doc.fontSize(14).text('SHIPPING LABEL', 100, 50);
    doc.fontSize(10);
    doc.text(`Order: ${order.orderNumber}`, 100, 80);
    doc.text(`Tracking: ${order.trackingNumber}`, 100, 100);
    doc.text(`Total Items: ${order.totalQty}`, 100, 120);
    doc.text(`Total Amount: $${order.totalAmount}`, 100, 140);

    // Barcode (যদি লাইব্রেরি ব্যবহার করেন)
    // doc.image('barcode.png', 100, 160);

    // S3-এ আপলোড করুন
    const buffer = doc.getStream().read();
    const url = await this.uploadToS3(fileName, buffer);

    return url;
  }

  // ইনভয়েস জেনারেট করুন
  async generateInvoice(order: any, wholesaler: any): Promise<string> {
    const doc = new PDFDocument();
    const fileName = `invoice-${order.id}.pdf`;

    // ইনভয়েস হেডার
    doc.fontSize(16).text('INVOICE', 50, 50);
    doc.fontSize(10);
    doc.text(`From: ${wholesaler.name}`, 50, 80);
    doc.text(`Email: ${wholesaler.email}`, 50, 100);

    // অর্ডার ডিটেইলস
    doc.fontSize(12).text('Order Details', 50, 140);
    doc.fontSize(10);
    doc.text(`Order Number: ${order.orderNumber}`, 50, 160);
    doc.text(`Date: ${new Date(order.createdAt).toLocaleDateString()}`, 50, 180);
    doc.text(`Total: $${order.totalAmount}`, 50, 200);

    // লাইন আইটেম
    doc.fontSize(10).text('Items:', 50, 240);
    let yPosition = 260;

    const items = order.items || [];
    for (const item of items) {
      doc.text(`- ${item.productId}: ${item.qty} x $${item.price}`, 50, yPosition);
      yPosition += 20;
    }

    // S3-এ আপলোড করুন
    const buffer = doc.getStream().read();
    const url = await this.uploadToS3(fileName, buffer);

    return url;
  }

  // প্যাকেজিং স্লিপ জেনারেট করুন
  async generatePackingSlip(order: any): Promise<string> {
    const doc = new PDFDocument();
    const fileName = `packing-slip-${order.id}.pdf`;

    doc.fontSize(14).text('PACKING SLIP', 50, 50);
    doc.fontSize(10);
    doc.text(`Order: ${order.orderNumber}`, 50, 80);
    doc.text(`Items to pack: ${order.totalQty}`, 50, 100);

    // আইটেম লিস্ট
    const items = order.items || [];
    let yPosition = 140;

    for (const item of items) {
      doc.text(`[ ] ${item.qty}x ${item.productId}`, 50, yPosition);
      yPosition += 20;
    }

    const buffer = doc.getStream().read();
    const url = await this.uploadToS3(fileName, buffer);

    return url;
  }

  private async uploadToS3(fileName: string, buffer: Buffer): Promise<string> {
    const key = `wholesaler-documents/${fileName}`;

    await this.s3
      .putObject({
        Bucket: process.env.AWS_S3_BUCKET || '',
        Key: key,
        Body: buffer,
        ContentType: 'application/pdf'
      })
      .promise();

    return `https://${process.env.AWS_S3_BUCKET}.s3.amazonaws.com/${key}`;
  }
}