import { Injectable } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class WholesalePricingService {
  constructor(private prisma: PrismaService) {}

  // টায়ার্ড প্রাইসিং রুল যোগ করুন
  async addPricingTier(
    wholesalerId: string,
    productId: string,
    tiers: Array<{
      minQty: number;
      maxQty?: number;
      discountPercent: number;
    }>
  ) {
    const product = await this.prisma.wholesaleProduct.findUnique({
      where: { id: productId }
    });

    if (!product) throw new Error('Product not found');

    // টায়ার্ড প্রাইসিং JSON হিসেবে সংরক্ষণ করুন
    const tierPricing: any = {};

    for (const tier of tiers) {
      const key = tier.maxQty ? `${tier.minQty}-${tier.maxQty}` : `${tier.minQty}+`;
      const price = product.basePrice * (1 - tier.discountPercent / 100);
      tierPricing[key] = price;
    }

    await this.prisma.wholesaleProduct.update({
      where: { id: productId },
      data: { tierPricing }
    });

    return { success: true };
  }

  // অর্ডার কোয়ান্টিটি অনুযায়ী প্রাইস পান
  getPriceForQuantity(product: any, quantity: number): number {
    const tierPricing = product.tierPricing || {};

    // টায়ার ম্যাচ করুন
    for (const [range, price] of Object.entries(tierPricing)) {
      if (range.includes('+')) {
        // "50+" ফরম্যাট
        const minQty = parseInt(range.split('+')[0]);
        if (quantity >= minQty) {
          return price as number;
        }
      } else if (range.includes('-')) {
        // "10-50" ফরম্যাট
        const [min, max] = range.split('-').map(Number);
        if (quantity >= min && quantity <= max) {
          return price as number;
        }
      }
    }

    // ডিফল্ট প্রাইস রিটার্ন করুন
    return Number(product.basePrice);
  }
}