import { Injectable, Logger } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';
import { Cron } from '@nestjs/schedule';
import axios from 'axios';

@Injectable()
export class StockSyncService {
  private logger = new Logger('StockSyncService');

  constructor(private prisma: PrismaService) {}

  // প্রতি 6 ঘণ্টায় স্টক সিঙ্ক করুন
  @Cron('0 */6 * * *')
  async syncWholesalerStock() {
    const syncs = await this.prisma.wholesalerStockSync.findMany({
      where: { status: { in: ['pending', 'completed'] } }
    });

    for (const sync of syncs) {
      await this.performStockSync(sync.id);
    }
  }

  async performStockSync(syncId: string) {
    const sync = await this.prisma.wholesalerStockSync.findUnique({
      where: { id: syncId }
    });

    if (!sync) return;

    try {
      await this.prisma.wholesalerStockSync.update({
        where: { id: syncId },
        data: { status: 'syncing' }
      });

      // হোলসেলার API থেকে স্টক ডেটা পান
      const stockData = await this.fetchWholesalerStock(sync.wholesalerId);

      // পণ্য স্টক আপডেট করুন
      for (const item of stockData) {
        await this.prisma.product.updateMany({
          where: {
            tenantId: sync.tenantId,
            sku: item.sku
          },
          data: {
            stock: item.stock,
            updated: new Date()
          }
        });
      }

      await this.prisma.wholesalerStockSync.update({
        where: { id: syncId },
        data: {
          status: 'completed',
          lastSyncedAt: new Date(),
          nextSyncAt: new Date(Date.now() + 6 * 60 * 60 * 1000),
          error: null
        }
      });

      this.logger.log(`Stock sync completed for wholesaler: ${sync.wholesalerId}`);
    } catch (err) {
      this.logger.error('Stock sync failed:', err);

      await this.prisma.wholesalerStockSync.update({
        where: { id: syncId },
        data: {
          status: 'failed',
          error: (err as Error).message
        }
      });
    }
  }

  private async fetchWholesalerStock(wholesalerId: string): Promise<any[]> {
    // বিভিন্ন হোলসেলার API থেকে স্টক ডেটা পান
    // এই উদাহরণ Merrono API-এর জন্য

    try {
      const response = await axios.get(
        `https://api.merrono.com/products/stock?wholesaler_id=${wholesalerId}`,
        {
          headers: { Authorization: `Bearer ${process.env.MERRONO_API_KEY}` }
        }
      );

      return response.data.items || [];
    } catch (err) {
      this.logger.error('Failed to fetch wholesaler stock:', err);
      return [];
    }
  }
}