import { Injectable, Logger, BadRequestException } from '@nestjs/common';
import { PrismaService } from '../../database/prisma.service';

@Injectable()
export class WholesalerService {
  private logger = new Logger('WholesalerService');

  constructor(private prisma: PrismaService) {}

  // নতুন হোলসেলার অ্যাকাউন্ট তৈরি করুন
  async createWholesalerAccount(input: {
    name: string;
    email: string;
    phone: string;
    planType: 'basic' | 'standard' | 'premium';
  }) {
    try {
      // টেন্যান্ট তৈরি করুন
      const tenant = await this.prisma.tenant.create({
        data: {
          name: input.name,
          email: input.email,
          type: 'wholesaler',
          status: 'ACTIVE',
          country: 'BD' // Default
        }
      });

      // সাবডোমেইন তৈরি করুন (auto-generate)
      const subdomain = this.generateSubdomain(input.name);

      // হোলসেলার অ্যাকাউন্ট তৈরি করুন
      const wholesaler = await this.prisma.wholesaler.create({
        data: {
          tenantId: tenant.id,
          name: input.name,
          email: input.email,
          phone: input.phone,
          subdomain,
          planType: input.planType,
          monthlyFee: this.getPlanFee(input.planType),
          status: 'active'
        }
      });

      // Analytics রেকর্ড তৈরি করুন
      await this.prisma.wholesaleAnalytics.create({
        data: {
          wholesalerId: wholesaler.id
        }
      });

      this.logger.log(`Wholesaler account created: ${wholesaler.id}`);

      return {
        success: true,
        data: {
          wholesalerId: wholesaler.id,
          subdomain: `${subdomain}.${process.env.APP_DOMAIN}`,
          storeUrl: `${subdomain}.${process.env.APP_DOMAIN}`
        }
      };
    } catch (err) {
      this.logger.error('Wholesaler creation failed:', err);
      throw err;
    }
  }

  // ব্র্যান্ডিং আপডেট করুন
  async updateBranding(wholesalerId: string, input: {
    logoUrl?: string;
    brandingColor?: string;
    customDomain?: string;
  }) {
    const wholesaler = await this.prisma.wholesaler.update({
      where: { id: wholesalerId },
      data: {
        logoUrl: input.logoUrl,
        brandingColor: input.brandingColor,
        customDomain: input.customDomain
      }
    });

    // কাস্টম ডোমেইনের জন্য SSL সেটআপ করুন
    if (input.customDomain) {
      await this.setupCustomDomainSSL(input.customDomain);
    }

    return { success: true, data: wholesaler };
  }

  // হোলসেলার সাইট সেটিংস পান
  async getWholesalerSite(wholesalerId: string) {
    const wholesaler = await this.prisma.wholesaler.findUnique({
      where: { id: wholesalerId }
    });

    return {
      success: true,
      data: {
        name: wholesaler?.name,
        logo: wholesaler?.logoUrl,
        primaryColor: wholesaler?.brandingColor,
        domain: wholesaler?.customDomain || `${wholesaler?.subdomain}.${process.env.APP_DOMAIN}`
      }
    };
  }

  private generateSubdomain(name: string): string {
    return name
      .toLowerCase()
      .replace(/[^\w\s-]/g, '')
      .replace(/\s+/g, '-')
      .substring(0, 20);
  }

  private getPlanFee(planType: string): number {
    const fees: any = {
      basic: 99,
      standard: 199,
      premium: 499
    };

    return fees[planType] || 99;
  }

  private async setupCustomDomainSSL(domain: string) {
    // cert-manager বা Let's Encrypt দিয়ে SSL সেটআপ করুন
    this.logger.log(`SSL setup initiated for domain: ${domain}`);
  }
}