import { Injectable } from '@nestjs/common';
import { Process, Processor } from '@nestjs/bull';
import { Job } from 'bull';
import * as nodemailer from 'nodemailer';

@Processor('email')
@Injectable()
export class EmailQueue {
  private transporter: any;

  constructor() {
    this.transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST,
      port: parseInt(process.env.SMTP_PORT || '587'),
      secure: process.env.SMTP_SECURE === 'true',
      auth: {
        user: process.env.SMTP_USER,
        pass: process.env.SMTP_PASS
      }
    });
  }

  @Process('send-verification')
  async sendVerificationEmail(job: Job) {
    const { email, token, tenantHost } = job.data;

    const verifyLink = `${process.env.APP_URL}/auth/verify?token=${token}`;

    await this.transporter.sendMail({
      from: `noreply@${tenantHost}`,
      to: email,
      subject: 'Verify your email',
      html: `
        <h2>Welcome!</h2>
        <p>Click the link below to verify your email:</p>
        <a href="${verifyLink}">Verify Email</a>
      `
    });

    return { success: true };
  }

  @Process('send-password-reset')
  async sendPasswordReset(job: Job) {
    const { email, token, tenantHost } = job.data;

    const resetLink = `${process.env.APP_URL}/auth/reset-password?token=${token}`;

    await this.transporter.sendMail({
      from: `noreply@${tenantHost}`,
      to: email,
      subject: 'Reset your password',
      html: `
        <h2>Password Reset</h2>
        <p>Click below to reset your password (valid for 30 minutes):</p>
        <a href="${resetLink}">Reset Password</a>
      `
    });

    return { success: true };
  }

  @Process('send-order-confirmation')
  async sendOrderConfirmation(job: Job) {
    const { email, orderNumber, totalAmount, tenantHost } = job.data;

    await this.transporter.sendMail({
      from: `orders@${tenantHost}`,
      to: email,
      subject: `Order Confirmed: ${orderNumber}`,
      html: `
        <h2>Order Confirmed</h2>
        <p>Order #${orderNumber}</p>
        <p>Total: $${totalAmount}</p>
      `
    });

    return { success: true };
  }
}