import React, { useState } from 'react';
import { DragDropContext, Droppable, Draggable } from 'react-beautiful-dnd';
import axios from 'axios';

export default function PageEditor({ pageId }: { pageId: string }) {
  const [blocks, setBlocks] = useState<any[]>([]);
  const [selectedBlock, setSelectedBlock] = useState<any>(null);

  // ব্লক যোগ করুন
  const addBlock = async (type: string) => {
    const token = localStorage.getItem('accessToken');
    const res = await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/builder/blocks`,
      {
        pageId,
        type,
        position: blocks.length,
        settings: {},
        content: ''
      },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setBlocks([...blocks, res.data.data]);
  };

  // ব্লক সেটিংস আপডেট করুন
  const updateBlock = async (blockId: string, settings: any) => {
    const token = localStorage.getItem('accessToken');
    await axios.put(
      `${process.env.NEXT_PUBLIC_API_URL}/builder/blocks/${blockId}`,
      { settings },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setBlocks(blocks.map((b) => (b.id === blockId ? { ...b, settings } : b)));
  };

  // ড্র্যাগ-ড্রপ হ্যান্ডলার
  const onDragEnd = async (result: any) => {
    const { source, destination } = result;

    if (!destination) return;

    const newBlocks = Array.from(blocks);
    const [movedBlock] = newBlocks.splice(source.index, 1);
    newBlocks.splice(destination.index, 0, movedBlock);

    setBlocks(newBlocks);

    // সার্ভারে সংরক্ষণ করুন
    const token = localStorage.getItem('accessToken');
    await axios.put(
      `${process.env.NEXT_PUBLIC_API_URL}/builder/pages/${pageId}`,
      { content: newBlocks },
      { headers: { Authorization: `Bearer ${token}` } }
    );
  };

  return (
    <div style={{ display: 'grid', gridTemplateColumns: '250px 1fr 300px' }}>
      {/* Block Palette */}
      <div style={{ background: '#f5f5f5', padding: '1rem' }}>
        <h3>Blocks</h3>
        <button onClick={() => addBlock('hero')}>+ Hero</button>
        <button onClick={() => addBlock('product')}>+ Product</button>
        <button onClick={() => addBlock('text')}>+ Text</button>
        <button onClick={() => addBlock('image')}>+ Image</button>
        <button onClick={() => addBlock('button')}>+ Button</button>
        <button onClick={() => addBlock('newsletter')}>+ Newsletter</button>
      </div>

      {/* Canvas */}
      <div style={{ background: '#fff', minHeight: '800px', padding: '2rem' }}>
        <DragDropContext onDragEnd={onDragEnd}>
          <Droppable droppableId="canvas">
            {(provided) => (
              <div
                {...provided.droppableProps}
                ref={provided.innerRef}
                style={{ minHeight: '800px' }}
              >
                {blocks.map((block, index) => (
                  <Draggable key={block.id} draggableId={block.id} index={index}>
                    {(provided, snapshot) => (
                      <div
                        ref={provided.innerRef}
                        {...provided.draggableProps}
                        {...provided.dragHandleProps}
                        style={{
                          background: snapshot.isDragging ? '#e3f2fd' : '#fff',
                          border: '2px solid #ddd',
                          padding: '1rem',
                          marginBottom: '1rem',
                          cursor: 'pointer'
                        }}
                        onClick={() => setSelectedBlock(block)}
                      >
                        <div style={{ display: 'flex', justifyContent: 'space-between' }}>
                          <span>{block.type}</span>
                          <button
                            onClick={async () => {
                              const token = localStorage.getItem('accessToken');
                              await axios.delete(
                                `${process.env.NEXT_PUBLIC_API_URL}/builder/blocks/${block.id}`,
                                { headers: { Authorization: `Bearer ${token}` } }
                              );
                              setBlocks(blocks.filter((b) => b.id !== block.id));
                            }}
                          >
                            ✕
                          </button>
                        </div>
                        <p>{block.content || '(empty)'}</p>
                      </div>
                    )}
                  </Draggable>
                ))}
                {provided.placeholder}
              </div>
            )}
          </Droppable>
        </DragDropContext>
      </div>

      {/* Settings Panel */}
      {selectedBlock && (
        <div style={{ background: '#f9f9f9', padding: '1rem', borderLeft: '1px solid #ddd' }}>
          <h3>Settings</h3>
          <label>
            Content:
            <textarea
              value={selectedBlock.content}
              onChange={(e) =>
                updateBlock(selectedBlock.id, { ...selectedBlock.settings, content: e.target.value })
              }
            />
          </label>

          {selectedBlock.type === 'button' && (
            <>
              <label>
                Button Text:
                <input
                  value={selectedBlock.settings.text || ''}
                  onChange={(e) => updateBlock(selectedBlock.id, { ...selectedBlock.settings, text: e.target.value })}
                />
              </label>
              <label>
                Link:
                <input
                  value={selectedBlock.settings.link || ''}
                  onChange={(e) => updateBlock(selectedBlock.id, { ...selectedBlock.settings, link: e.target.value })}
                />
              </label>
            </>
          )}

          <button>Save</button>
        </div>
      )}
    </div>
  );
}