import React, { useEffect, useState } from 'react';
import axios from 'axios';
import i18n from 'i18next';

export default function LanguageSwitcher() {
  const [language, setLanguage] = useState('en');
  const [countryCode, setCountryCode] = useState('US');

  useEffect(() => {
    // ব্যবহারকারীর লোকেশন ডিটেক্ট করুন
    const detectLocation = async () => {
      try {
        // IP Geolocation API ব্যবহার করুন
        const res = await axios.get('https://ipapi.co/json/');
        
        const code = res.data.country_code;
        setCountryCode(code);

        // দেশের জন্য ডিফল্ট ভাষা সেট করুন
        const defaultLanguage = getLanguageForCountry(code);
        setLanguage(defaultLanguage);

        // i18n ভাষা পরিবর্তন করুন
        i18n.changeLanguage(defaultLanguage);

        // লোকাল স্টোরেজ সংরক্ষণ করুন
        localStorage.setItem('preferredLanguage', defaultLanguage);
        localStorage.setItem('countryCode', code);
      } catch (err) {
        console.error('Location detection failed:', err);
      }
    };

    // ব্রাউজার ভাষা পরিবর্তন শুনুন
    const handleLanguageChange = (language: string) => {
      setLanguage(language);
      i18n.changeLanguage(language);
      localStorage.setItem('preferredLanguage', language);
    };

    detectLocation();

    return () => {
      // Cleanup
    };
  }, []);

  const handleLanguageSelect = (lang: string) => {
    setLanguage(lang);
    i18n.changeLanguage(lang);
    localStorage.setItem('preferredLanguage', lang);
  };

  return (
    <div className="language-switcher">
      <select value={language} onChange={(e) => handleLanguageSelect(e.target.value)}>
        <option value="en">English</option>
        <option value="bn">বাংলা</option>
        <option value="hi">हिन्दी</option>
        <option value="ur">اردو</option>
        <option value="ta">தமிழ்</option>
      </select>

      <span>📍 {countryCode}</span>
    </div>
  );
}

function getLanguageForCountry(countryCode: string): string {
  const languageMap: any = {
    BD: 'bn',
    IN: 'hi',
    US: 'en',
    GB: 'en',
    PK: 'ur',
    LK: 'ta',
    // Add more mappings
  };

  return languageMap[countryCode] || 'en';
}