import React, { useEffect, useState } from 'react';
import axios from 'axios';
import styles from '../../styles/OrderTracking.module.css';

export default function OrderTracking({ orderId }: { orderId: string }) {
  const [order, setOrder] = useState<any>(null);
  const [tracking, setTracking] = useState<any[]>([]);

  useEffect(() => {
    const ws = new WebSocket(
      `${process.env.NEXT_PUBLIC_WS_URL}/orders/${orderId}/tracking`
    );

    ws.onmessage = (event) => {
      const data = JSON.parse(event.data);
      setOrder(data.order);
      setTracking(data.timeline);
    };

    return () => ws.close();
  }, [orderId]);

  const statusSteps = [
    { status: 'PENDING', label: 'Order Placed', icon: '📦' },
    { status: 'PROCESSING', label: 'Processing', icon: '⚙️' },
    { status: 'SHIPPED', label: 'Shipped', icon: '🚚' },
    { status: 'DELIVERED', label: 'Delivered', icon: '✅' }
  ];

  const currentStepIndex = statusSteps.findIndex((s) => s.status === order?.status);

  return (
    <div className={styles.container}>
      <h2>Order Tracking</h2>

      {/* প্রগ্রেস বার */}
      <div className={styles.progressBar}>
        {statusSteps.map((step, index) => (
          <div
            key={step.status}
            className={`${styles.step} ${index <= currentStepIndex ? styles.completed : ''}`}
          >
            <div className={styles.icon}>{step.icon}</div>
            <p>{step.label}</p>
          </div>
        ))}
      </div>

      {/* টাইমলাইন */}
      <div className={styles.timeline}>
        {tracking.map((event, index) => (
          <div key={index} className={styles.timelineItem}>
            <div className={styles.marker}></div>
            <div className={styles.content}>
              <h4>{event.status}</h4>
              <p>{event.message}</p>
              <small>{new Date(event.timestamp).toLocaleString()}</small>
            </div>
          </div>
        ))}
      </div>

      {/* রিয়েল-টাইম আপডেট */}
      {order?.trackingNumber && (
        <div className={styles.trackingInfo}>
          <p>Tracking Number: <strong>{order.trackingNumber}</strong></p>
          <p>Courier: <strong>{order.courier}</strong></p>
        </div>
      )}
    </div>
  );
}