import React, { useState, useEffect } from 'react';
import axios from 'axios';
import styles from '../../styles/QASection.module.css';

export default function QASection({ productId }: { productId: string }) {
  const [qnas, setQnas] = useState<any[]>([]);
  const [newQuestion, setNewQuestion] = useState('');
  const [showForm, setShowForm] = useState(false);

  useEffect(() => {
    fetchQnA();
  }, [productId]);

  const fetchQnA = async () => {
    const res = await axios.get(
      `${process.env.NEXT_PUBLIC_API_URL}/products/${productId}/qa`
    );
    setQnas(res.data.qnas);
  };

  const submitQuestion = async () => {
    const token = localStorage.getItem('accessToken');

    await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/products/${productId}/qa`,
      { question: newQuestion },
      { headers: { Authorization: `Bearer ${token}` } }
    );

    setNewQuestion('');
    setShowForm(false);
    fetchQnA();
  };

  const markHelpful = async (qaId: string) => {
    await axios.post(
      `${process.env.NEXT_PUBLIC_API_URL}/qa/${qaId}/helpful`
    );
    fetchQnA();
  };

  return (
    <div className={styles.container}>
      <h3>Customer Questions & Answers</h3>

      {!showForm ? (
        <button onClick={() => setShowForm(true)}>Ask a Question</button>
      ) : (
        <div className={styles.form}>
          <textarea
            placeholder="Ask your question..."
            value={newQuestion}
            onChange={(e) => setNewQuestion(e.target.value)}
          />
          <button onClick={submitQuestion}>Submit</button>
          <button onClick={() => setShowForm(false)}>Cancel</button>
        </div>
      )}

      {/* Q&A লিস্ট */}
      <div className={styles.qaList}>
        {qnas.map((qa) => (
          <div key={qa.id} className={styles.qaItem}>
            <div className={styles.question}>
              <p>
                <strong>Q: {qa.question}</strong>
              </p>
              <small>Asked by {qa.askedBy}</small>
            </div>

            {qa.answer && (
              <div className={styles.answer}>
                <p>
                  <strong>A: {qa.answer}</strong>
                </p>
              </div>
            )}

            <button onClick={() => markHelpful(qa.id)}>👍 Helpful ({qa.upvotes})</button>
          </div>
        ))}
      </div>
    </div>
  );
}